/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   FreeBSD/OpenBSD/NetBSD CDDA read ioctl support
 */
#ifndef LINT
static char *_fbread_c_ident_ = "@(#)fbread.c	7.14 02/04/09";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_RD_FB) && defined(CDDA_SYSVIPC)

#include <sys/cdio.h>
#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/fbread.h"


#ifndef CDIOCREADAUDIO
#define CDIOCREADAUDIO	_IOWR('c',31,struct ioc_read_audio)

struct ioc_read_audio  
{
	u_char		address_format;
	union msf_lba	address;
	int		nframes;
	u_char		*buffer;
};
#endif	/* CDIOCREADAUDIO */


extern appdata_t	app_data;
extern FILE		*errfp;
extern void		*cdda_shmaddr;

STATIC int		fb_rsemid;	/* Semaphores identifier */
STATIC cd_state_t	*fb_rcd;	/* CD state pointer */


/*
 * fbcd_read_fillbuf
 *	Before beginning to play we fill the data buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
fbcd_read_fillbuf(int fd)
{
	int			i;
	byte_t			*start,
				*end,
				*p;
	int			lba,
				nframes;
	struct ioc_read_audio	cdda;

	/* Get lock */
	cdda_waitsem(fb_rsemid, LOCK);

	/* Initialize */
	start = &fb_rcd->cdb->data[fb_rcd->cds->olap_bytes];

	/* Set up for read */
	lba = fb_rcd->i->start_lba;
	nframes = fb_rcd->cds->chunk_blocks + fb_rcd->cds->olap_blocks;
	cdda.address.lba = util_bswap32(lba);
	cdda.address_format = CD_LBA_FORMAT;
	cdda.nframes = util_bswap32(nframes);
	cdda.buffer = (u_char *) start;

	cdda_rd_setcurtrk(fb_rcd, (int) cdda.address.lba);

	/* Fill up our buffer */
	i = 0;

	while (i < fb_rcd->cds->buffer_chunks && !fb_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * fb_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (lba + fb_rcd->cds->chunk_blocks >= fb_rcd->i->end_lba) {
			nframes = fb_rcd->i->end_lba - lba;
			cdda.nframes = util_bswap32(nframes);
			(void) memset(
				fb_rcd->cdb->data,
				0,
				fb_rcd->cds->chunk_bytes +
				(fb_rcd->cds->olap_bytes << 1)
			);
			fb_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDIOCREADAUDIO, &cdda) < 0) {
			(void) sprintf(fb_rcd->i->msgbuf,
					"fbcd_read_fillbuf: CDIOCREADAUDIO "
					"ioctl failed: %s",
					strerror(errno));
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", fb_rcd->i->msgbuf);
			cdda_postsem(fb_rsemid, LOCK);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
			"\nSent CDIOCREADAUDIO ioctl: blk=%d len=%d\n",
			lba, nframes);

		/* Do jitter correction */
		if (i == 0)
			p = &fb_rcd->cdb->data[fb_rcd->cds->olap_bytes];
		else
			p = cdda_rd_corrjitter(fb_rcd);

		/* Data end */
		end = p + fb_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (fb_rcd->i->jitter) {
			(void) memcpy(
				fb_rcd->cdb->olap,
				end - fb_rcd->cds->search_bytes,
				fb_rcd->cds->search_bytes << 1
			);
		}

		/* Copy in */
		(void) memcpy(
			&fb_rcd->cdb->b[
			    fb_rcd->cds->chunk_bytes * fb_rcd->cdb->nextin
			],
			p,
			fb_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		fb_rcd->cdb->nextin++;
		fb_rcd->cdb->nextin %= fb_rcd->cds->buffer_chunks;
		fb_rcd->cdb->occupied++;

		lba += ((end - start) / CDDA_BLKSZ);
		cdda.address.lba = util_bswap32(lba);

		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1)) {
			lba++;
			cdda.address.lba = util_bswap32(lba);
		}

		cdda_rd_setcurtrk(fb_rcd, (int) cdda.address.lba);

		i++;

		/* Update debug level */
		app_data.debug = fb_rcd->i->debug;
	}

	/* No room available now */
	cdda_waitsem(fb_rsemid, ROOM);

	/* Signal data available */
	cdda_postsem(fb_rsemid, DATA);

	/* Release lock */
	cdda_postsem(fb_rsemid, LOCK);

	return TRUE;
}


/*
 * fbcd_read_cont
 *	Function responsible for reading from the cd,
 *	correcting for jitter and writing to the buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
fbcd_read_cont(int fd)
{
	byte_t			*start,
				*end,
				*p;
	int			lba,
				nframes;
	struct ioc_read_audio	cdda;

	/* Initialize */
	start = &fb_rcd->cdb->data[fb_rcd->cds->olap_bytes];

	/* Set up for read */
	lba = fb_rcd->i->start_lba + fb_rcd->cds->buffer_blocks;
	nframes = fb_rcd->cds->chunk_blocks + fb_rcd->cds->olap_blocks;
	cdda.address.lba = util_bswap32(lba);
	cdda.address_format = CD_LBA_FORMAT;
	cdda.nframes = util_bswap32(nframes);
	cdda.buffer = (u_char *) start;

	cdda_rd_setcurtrk(fb_rcd, cdda.address.lba);

	/* While not stopped or finished */
	while (fb_rcd->i->state != CDSTAT_COMPLETED && !fb_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * fb_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (lba + fb_rcd->cds->chunk_blocks >= fb_rcd->i->end_lba) {
			nframes = fb_rcd->i->end_lba - lba;
			cdda.nframes = util_bswap32(nframes);
			(void) memset(
				fb_rcd->cdb->data,
				0,
				fb_rcd->cds->chunk_bytes +
				(fb_rcd->cds->olap_bytes << 1)
			);
			fb_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDIOCREADAUDIO, &cdda) < 0) {
			(void) sprintf(fb_rcd->i->msgbuf,
					"fbcd_read_cont: CDIOCREADAUDIO ioctl "
					"failed: %s",
					strerror(errno));
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
			"\nSent CDIOCREADAUDIO ioctl: blk=%d len=%d\n",
			lba, nframes);

		/* Do jitter correction */
		p = cdda_rd_corrjitter(fb_rcd);

		/* Data end */
		end = p + fb_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (fb_rcd->i->jitter) {
			(void) memcpy(
				fb_rcd->cdb->olap,
				end - fb_rcd->cds->search_bytes,
				fb_rcd->cds->search_bytes << 1
			);
		}

		/* Get lock */
		cdda_waitsem(fb_rsemid, LOCK);

		/* Wait until there is room */
		while (fb_rcd->cdb->occupied >= fb_rcd->cds->buffer_chunks &&
		       fb_rcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(fb_rsemid, LOCK);
			cdda_waitsem(fb_rsemid, ROOM);
			cdda_waitsem(fb_rsemid, LOCK);
		}

		/* Break if completed */
		if (fb_rcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(fb_rsemid, LOCK);
			break;
		}

		/* Copy to free location */
		(void) memcpy(
			&fb_rcd->cdb->b[
			    fb_rcd->cds->chunk_bytes * fb_rcd->cdb->nextin
			],
			p,
			fb_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		fb_rcd->cdb->nextin++;
		fb_rcd->cdb->nextin %= fb_rcd->cds->buffer_chunks;
		fb_rcd->cdb->occupied++;

		lba += ((end - start) / CDDA_BLKSZ);
		cdda.address.lba = util_bswap32(lba);

		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1)) {
			lba++;
			cdda.address.lba = util_bswap32(lba);
		}

		cdda_rd_setcurtrk(fb_rcd, cdda.address.lba);

		/* Signal data available */
		cdda_postsem(fb_rsemid, DATA);

		/* Release lock */
		cdda_postsem(fb_rsemid, LOCK);

		/* Update debug level */
		app_data.debug = fb_rcd->i->debug;
	}

	/* Wake up writer */
	cdda_postsem(fb_rsemid, DATA);

	return TRUE;
}


/*
 * fbcd_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killwriter - whether to kill the writer process
 *
 * Return:
 *	Nothing.
 */
STATIC void
fbcd_cleanup(bool_t killwriter)
{
	DBGPRN(DBG_DEVIO)(errfp,
			  "\nfbcd_cleanup: Cleaning up reader pid=%d\n",
			  (int) getpid());

	(void) sleep(1);

	if (fb_rcd != NULL) {
		if (killwriter && fb_rcd->i != NULL && fb_rcd->i->writer > 0) {
			(void) kill(fb_rcd->i->writer, SIGTERM);
			fb_rcd->i->writer = 0;
			fb_rcd->i->state = CDSTAT_COMPLETED;
		}

		MEM_FREE(fb_rcd);
		fb_rcd = NULL;
	}
}


/*
 * fbcd_onterm
 *	SIGTERM signal handler - gracefully exit
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
fbcd_onterm(int signo)
{
	fbcd_cleanup(FALSE);
	exit(1);
}


/*
 * fb_readinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
fb_readinit(void)
{
	return CDDA_READAUDIO;
}


/*
 * fb_read
 *	Attaches to shared memory and semaphores. Continuously reads
 *	data from the cd and places into shared memory.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
fb_read(int fd)
{
	struct sigaction	action_term;

	/* Check configuration */
	if (app_data.di_method != 2 /* DI_FBIOC */) {
		(void) fprintf(errfp,
			"fb_read: Inconsistent deviceInterfaceMethod "
			"and cddaReadMethod parameters.  Aborting.\n");
		return FALSE;
	}

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);

	action_term.sa_handler = fbcd_onterm;
	sigemptyset(&action_term.sa_mask);
	action_term.sa_flags = 0;
	if (sigaction(SIGTERM, &action_term, NULL) < 0) {
		(void) fprintf(errfp,
				"fb_read: sigaction failed (SIGTERM): %s\n",
				strerror(errno));
		return FALSE;
	}

	/* Allocate mempry */
	fb_rcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (fb_rcd == NULL) {
		(void) fprintf(errfp, "fb_read: out of memory\n");
		return FALSE;
	}

	(void) memset(fb_rcd, 0, sizeof(cd_state_t));

	/* Initialize cd pointers to point into shared memory */
	if ((fb_rsemid = cdda_rw_initipc(fb_rcd)) < 0) {
		fbcd_cleanup(TRUE);
		return FALSE;
	}

	DBGPRN(DBG_DEVIO)(errfp,
			  "\nfbcd_read: Reading CDDA: "
			  "chunk_blks=%d, olap_blks=%d\n",
			  fb_rcd->cds->chunk_blocks, fb_rcd->cds->olap_blocks);

	/* Fill data buffer */
	if (!fbcd_read_fillbuf(fd)) {
		fbcd_cleanup(TRUE);
		return FALSE;
	}

	/* Keep reading */
	if (!fbcd_read_cont(fd)) {
		fbcd_cleanup(TRUE);
		return FALSE;
	}

	fbcd_cleanup(FALSE);
	return TRUE;
}

#endif	/* CDDA_RD_FB CDDA_SYSVIPC */

