/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   File-only (no audio playback) write method
 */
#ifndef LINT
static char *_flwrite_c_ident_ = "@(#)flwrite.c	7.31 02/04/24";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_WR_FILE) && defined(CDDA_SYSVIPC)

#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/genwrite.h"
#include "cdda_d/flwrite.h"


extern appdata_t	app_data;
extern FILE		*errfp;

STATIC int		fl_file_fd = -1,	/* Output file fd */
			fl_pipe_fd = -1,	/* Pipe to program fd */
			fl_wsemid;		/* Semaphores identifier */
STATIC pid_t		fl_pipe_pid = -1;	/* Pipe to program pid */
STATIC cd_state_t	*fl_wcd;		/* Shared memory pointer */
STATIC char		*fl_pipeprog;		/* Pipe to program */
STATIC bool_t		fl_write_file = FALSE,	/* Write to output file */
			fl_write_pipe = FALSE,	/* Pipe to program */
			fl_file_be = FALSE;	/* Big endian output file */
STATIC byte_t		fl_file_fmt;		/* Output file format */

/*
 * flaud_write
 *	Function responsible for writing from the buffer to the audio
 *	file.
 *
 * Args:
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
flaud_write(curstat_t *s)
{
	byte_t	*in_data,
		*sw_data,
		*le_data,
		*be_data,
		*file_data,
		*pipe_data;
	char	*fname = NULL;
	int	trk_idx = -1;
	bool_t	ret;

	/* Get memory for audio write buffer */
	le_data = (byte_t *) MEM_ALLOC("le_data", fl_wcd->cds->chunk_bytes);
	if (le_data == NULL) {
		(void) sprintf(fl_wcd->i->msgbuf,
				"flaud_write: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", fl_wcd->i->msgbuf);
		return FALSE;
	}

	/* Get memory for audio file write buffer */
	be_data = (byte_t *) MEM_ALLOC("be_data", fl_wcd->cds->chunk_bytes);
	if (be_data == NULL) {
		(void) sprintf(fl_wcd->i->msgbuf,
				"flaud_write: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", fl_wcd->i->msgbuf);
		return FALSE;
	}

	(void) memset(le_data, 0, fl_wcd->cds->chunk_bytes);
	(void) memset(be_data, 0, fl_wcd->cds->chunk_bytes);

	in_data = app_data.cdda_bigendian ? be_data : le_data;
	sw_data = app_data.cdda_bigendian ? le_data : be_data;

	/* While not stopped */
	ret = TRUE;
	while (fl_wcd->i->state != CDSTAT_COMPLETED) {
		/* Get lock */
		cdda_waitsem(fl_wsemid, LOCK);

		/* End if finished and nothing in the buffer */
		if (fl_wcd->cdb->occupied == 0 && fl_wcd->i->cdda_done) {
			fl_wcd->i->state = CDSTAT_COMPLETED;
			cdda_postsem(fl_wsemid, LOCK);
			break;
		}

		/* Wait until there is something there */
		while (fl_wcd->cdb->occupied <= 0 &&
		       fl_wcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(fl_wsemid, LOCK);
			cdda_waitsem(fl_wsemid, DATA);
			cdda_waitsem(fl_wsemid, LOCK);
		}

		/* Break if stopped */
		if (fl_wcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(fl_wsemid, LOCK);
			break;
		}

		/* Copy a chunk from the nextout slot into the data buffers */
		(void) memcpy(
			in_data,
			&fl_wcd->cdb->b[
			    fl_wcd->cds->chunk_bytes * fl_wcd->cdb->nextout
			],
			fl_wcd->cds->chunk_bytes
		);

		/* Update pointers */
		fl_wcd->cdb->nextout++;
		fl_wcd->cdb->nextout %= fl_wcd->cds->buffer_chunks;
		fl_wcd->cdb->occupied--;

		/* Signal room available */
		cdda_postsem(fl_wsemid, ROOM);

		/* Release lock */
		cdda_postsem(fl_wsemid, LOCK);

		/* Change channel routing if necessary */
		gen_chroute(fl_wcd->i->chroute, (sword16_t *)(void *) in_data,
			    (size_t) fl_wcd->cds->chunk_bytes);

		/* Prepare byte-swapped version of the data */
		gen_byteswap(in_data, sw_data,
			     (size_t) fl_wcd->cds->chunk_bytes);

		/* Write to file */
		if (fl_write_file) {
			file_data = fl_file_be ? be_data : le_data;

			/* Open new track file if necessary */
			if (!app_data.cdda_trkfile) {
				fname = s->trkinfo[0].outfile;
			}
			else if (trk_idx != fl_wcd->i->trk_idx) {
				trk_idx = fl_wcd->i->trk_idx;

				if (fl_file_fd >= 0)
					gen_close_file(
						fl_wcd,
						fl_file_fd,
						fl_file_fmt
					);

				fname = s->trkinfo[trk_idx].outfile;
				fl_file_fd = gen_open_file(
					fl_wcd,
					fname,
					O_WRONLY | O_TRUNC | O_CREAT,
					S_IRUSR | S_IWUSR,
					fl_file_fmt,
					fl_wcd->i->trk_len
				);
				if (fl_file_fd < 0) {
					fl_wcd->i->state = CDSTAT_COMPLETED;
					cdda_postsem(fl_wsemid, LOCK);
					ret = FALSE;
					break;
				}

				DBGPRN(DBG_DEVIO)(errfp,
					  "\nflaud_write: Writing file [%s]: "
					  "chunk_blks=%d\n",
					  fname, fl_wcd->cds->chunk_blocks);
			}

			if (!gen_write_chunk(fl_wcd, fl_file_fd,
					fname, file_data,
					(size_t) fl_wcd->cds->chunk_bytes)) {
				fl_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(fl_wsemid, LOCK);
				ret = FALSE;
				break;
			}
		}

		/* Pipe to program */
		if (fl_write_pipe) {
			pipe_data = fl_file_be ? be_data : le_data;

			if (!gen_write_chunk(fl_wcd, fl_pipe_fd,
					     fl_pipeprog, pipe_data,
					(size_t) fl_wcd->cds->chunk_bytes)) {
				fl_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(fl_wsemid, LOCK);
				ret = FALSE;
				break;
			}
		}

		fl_wcd->i->frm_played += fl_wcd->cds->chunk_blocks;

		while (fl_wcd->i->state == CDSTAT_PAUSED)
			util_delayms(400);

		/* Update debug level */
		app_data.debug = fl_wcd->i->debug;
	}

	/* Wake up reader */
	cdda_postsem(fl_wsemid, ROOM);

	/* Free memory */
	MEM_FREE(le_data);
	MEM_FREE(be_data);

	return (ret);
}


/*
 * flaud_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killreader - Whether to kill the reader process
 *
 * Return:
 *	Nothing.
 */
STATIC void
flaud_cleanup(bool_t killreader)
{
	DBGPRN(DBG_DEVIO)(errfp,
			  "\nflaud_cleanup: Cleaning up writer pid=%d\n",
			  (int) getpid());

	if (fl_file_fd >= 0) {
		gen_close_file(fl_wcd, fl_file_fd, fl_file_fmt);
		fl_file_fd = -1;
	}

	if (fl_pipe_fd >= 0) {
		gen_close_pipe(fl_wcd, fl_pipe_fd);
		fl_pipe_fd = -1;
	}

	(void) sleep(1);

	if (fl_pipe_pid > 0) {
		(void) kill(fl_pipe_pid, SIGTERM);
		fl_pipe_pid = -1;
	}

	if (fl_wcd != NULL) {
		if (fl_wcd->i != NULL) {
			if (killreader && fl_wcd->i->reader > 0) {
				(void) kill(fl_wcd->i->reader, SIGTERM);
				fl_wcd->i->reader = 0;
				fl_wcd->i->state = CDSTAT_COMPLETED;
			}

			/* Reset frames played counter */
			fl_wcd->i->frm_played = 0;
		}

		MEM_FREE(fl_wcd);
		fl_wcd = NULL;
	}
}


/*
 * flaud_onterm
 *	SIGTERM and SIGPIPE signal handler - gracefully exit
 *
 * Args:
 *	signo - The signal number
 *
 * Return:
 *	Nothing.
 */
STATIC void
flaud_onterm(int signo)
{
	flaud_cleanup(signo == SIGPIPE);
	exit(1);
}


/*
 * fl_writeinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
fl_writeinit(void)
{
	return (CDDA_WRITEFILE | CDDA_WRITEPIPE);
}


/*
 * fl_write
 *	Opens audio file, attaches shared memory and semaphores.
 *	Continuously reads data from shared memory and writes it
 *	to the audio file.
 *
 * Args:
 *	mode - play mode (bitmask PLAYMODE_CDDA, etc.)
 *	fmt  - output file format
 *	s    - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
fl_write(byte_t mode, byte_t fmt, curstat_t *s)
{
	int		ret = 0,
			wstat;
	word32_t	estlen;

	if (!PLAYMODE_IS_CDDA(mode)) {
		(void) fprintf(errfp, "fl_write: Nothing to do.\n");
		return FALSE;
	}

	if ((mode & PLAYMODE_FILE) != 0) {
		fl_write_file = TRUE;
	}
	if ((mode & PLAYMODE_PIPE) != 0) {
		fl_write_pipe = TRUE;
		fl_pipeprog = s->pipeprog;
	}
	fl_file_fmt = fmt;

	/* Generic services initialization */
	gen_write_init();

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);
	(void) signal(SIGALRM, SIG_IGN);
	(void) signal(SIGUSR1, SIG_IGN);
	(void) signal(SIGPIPE, flaud_onterm);
	(void) signal(SIGTERM, flaud_onterm);

	/* Allocate memory */
	fl_wcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (fl_wcd == NULL) {
		(void) fprintf(errfp, "fl_write: out of memory\n");
		return FALSE;
	}

	(void) memset(fl_wcd, 0, sizeof(cd_state_t));

	/* Initialize shared memory and semaphores */
	if ((fl_wsemid = cdda_rw_initipc(fl_wcd)) < 0) {
		flaud_cleanup(TRUE);
		return FALSE;
	}

	estlen = (fl_wcd->i->end_lba - fl_wcd->i->start_lba + 1) * CDDA_BLKSZ;

	if (fl_write_file || fl_write_pipe) {
		/* Open file and/or pipe */
		if (!app_data.cdda_trkfile && fl_write_file) {
			fl_file_fd = gen_open_file(
				fl_wcd,
				s->trkinfo[0].outfile,
				O_WRONLY | O_TRUNC | O_CREAT,
				S_IRUSR | S_IWUSR,
				fl_file_fmt, estlen
			);
			if (fl_file_fd < 0) {
				flaud_cleanup(TRUE);
				return FALSE;
			}

			DBGPRN(DBG_DEVIO)(errfp,
					  "\nflaud_write: Writing file [%s]: "
					  "chunk_blks=%d\n",
					  s->trkinfo[0].outfile,
					  fl_wcd->cds->chunk_blocks);
		}

		if (fl_write_pipe) {
			fl_pipe_fd = gen_open_pipe(fl_wcd, fl_pipeprog,
						   &fl_pipe_pid,
						   fl_file_fmt, estlen);
			if (fl_pipe_fd < 0) {
				flaud_cleanup(TRUE);
				return FALSE;
			}
			DBGPRN(DBG_DEVIO)(errfp,
					  "\nflaud_write: Pipe to [%s]: "
					  "chunk_blks=%d\n",
					  fl_pipeprog,
					  fl_wcd->cds->chunk_blocks);
		}

		switch ((int) fl_file_fmt) {
		case FILEFMT_AU:
		case FILEFMT_AIFF:
		case FILEFMT_AIFC:
			fl_file_be = TRUE;
			break;
		case FILEFMT_WAV:
		default:
			fl_file_be = FALSE;
			break;
		}
	}

	/* Do the writing */
	if (!flaud_write(s)) {
		flaud_cleanup(TRUE);
		return FALSE;
	}

	if (fl_write_file) {
		gen_close_file(fl_wcd, fl_file_fd, fl_file_fmt);
		fl_file_fd = -1;
	}

	if (fl_write_pipe) {
		gen_close_pipe(fl_wcd, fl_pipe_fd);
		fl_pipe_fd = -1;

		while ((ret = WAITPID(fl_pipe_pid, &wstat, 0)) != fl_pipe_pid){
			if (ret < 0 && errno != EINTR)
				break;
		}

		if (WIFEXITED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp,
					  "Pipe program exited, status %d\n",
					  WEXITSTATUS(wstat));
		}
		else if (WIFSIGNALED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp,
					  "Pipe program killed, signal %d\n",
					  WTERMSIG(wstat));
		}

		fl_pipe_pid = -1;
	}

	flaud_cleanup((bool_t) (ret != 0));
	return TRUE;
}

#endif	/* CDDA_WR_FILE CDDA_SYSVIPC */

