/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef LINT
static char *_genwrite_c_ident_ = "@(#)genwrite.c	7.22 02/04/24";
#endif

#include "common_d/appenv.h"
#include "common_d/patchlevel.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_SYSVIPC)

#include "cdda_d/genwrite.h"
#include "cdda_d/au.h"
#include "cdda_d/wav.h"
#include "cdda_d/aiff.h"


/* Defines used by gen_open_pipe() */
#define STR_SHPATH	"/bin/sh"	/* Path to shell */
#define STR_SHNAME	"sh"		/* Name of shell */
#define STR_SHARG	"-c"		/* Shell arg */


extern appdata_t	app_data;
extern FILE		*errfp;

STATIC uid_t		ruid,			/* Real user id */
			euid;			/* Effective user id */
STATIC gid_t		rgid,			/* Real group id */
			egid;			/* Effective group id */
STATIC char		*comment =
			"Output by xmcd " VERSION_MAJ "." VERSION_MIN;
						/* File comment */


/*
 * gen_set_eid
 *	Set effective uid/gid to that of the invoking user.  This is
 *	called before accessing any device or files for security reasons.
 *
 * Args:
 *	cdp - Pointer to the cd_state_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
gen_set_eid(cd_state_t *cdp)
{
#ifdef HAS_SETEUID
	if (seteuid(ruid) < 0 || geteuid() != ruid) {
		(void) sprintf(cdp->i->msgbuf, "gen_set_eid: Cannot set uid");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
		return FALSE;
	}
	if (setegid(rgid) < 0 || getegid() != rgid) {
		(void) sprintf(cdp->i->msgbuf, "gen_set_eid: Cannot set gid");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
		return FALSE;
	}
#endif
	return TRUE;
}


/*
 * gen_reset_eid
 *	Restore saved uid/gid after the use of gen_set_eid()..
 *
 * Args:
 *	cdp - Pointer to the cd_state_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
gen_reset_eid(cd_state_t *cdp)
{
#ifdef HAS_SETEUID
	if (seteuid(euid) < 0 || geteuid() != euid) {
		(void) sprintf(cdp->i->msgbuf,
				"gen_reset_eid: Cannot set uid");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
		return FALSE;
	}
	if (setegid(egid) < 0 || getegid() != egid) {
		(void) sprintf(cdp->i->msgbuf,
				"gen_reset_eid: Cannot set gid");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
		return FALSE;
	}
#endif

	return TRUE;
}


/*
 * gen_open_file
 *	Open an audio output file.
 *
 * Args:
 *	cdp	- Pointer to the cd_state_t structure
 *	path	- File path name
 *	oflag	- Open flags
 *	mode	- Open mode
 *	fmt	- File header format to be written
 *	datalen	- Audio data length in bytes
 *
 * Return:
 *	The file descriptor, or -1 if failed.
 */
int
gen_open_file(cd_state_t *cdp, char *path, int oflag, mode_t mode,
	      byte_t fmt, word32_t datalen)
{
	int	fd;

	if (!gen_set_eid(cdp))
		return FALSE;

	/* Open file */
	if ((fd = open(path, oflag, mode)) < 0) {
		(void) sprintf(cdp->i->msgbuf,
			    "gen_open_file: open of %s failed (errno=%d)",
			    path, errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
		(void) gen_reset_eid(cdp);
		return -1;
	}

	/* Update file header */
	switch ((int) fmt) {
	case FILEFMT_AU:
		if (!gen_write_au_hdr(cdp, fd, datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	case FILEFMT_WAV:
		if (!gen_write_wav_hdr(cdp, fd, datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	case FILEFMT_AIFF:
		if (!gen_write_aiff_hdr(cdp, fd, datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	case FILEFMT_AIFC:
		if (!gen_write_aifc_hdr(cdp, fd, datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	default:
		break;
	}

	(void) gen_reset_eid(cdp);
	return (fd);
}


/*
 * gen_close_file
 *	Close the audio output file.
 *
 * Args:
 *	cdp - Pointer to the cd_state_t structure
 *	fd  - The file descriptor
 *	fmt - File header format to be written
 *
 * Return:
 *	Nothing.
 */
void
gen_close_file(cd_state_t *cdp, int fd, byte_t fmt)
{
	if (!gen_set_eid(cdp))
		return;

	switch ((int) fmt) {
	case FILEFMT_AU:
		(void) gen_write_au_hdr(cdp, fd, 0);
		break;

	case FILEFMT_WAV:
		(void) gen_write_wav_hdr(cdp, fd, 0);
		break;

	case FILEFMT_AIFF:
		(void) gen_write_aiff_hdr(cdp, fd, 0);
		break;

	case FILEFMT_AIFC:
		(void) gen_write_aifc_hdr(cdp, fd, 0);
		break;

	default:
		break;
	}

	/* Close file */
	if (close(fd) < 0) {
		DBGPRN(DBG_DEVIO)(errfp,
			"gen_close_file: close failed (errno=%d)\n",
			errno);
	}

	(void) gen_reset_eid(cdp);
}


/*
 * gen_open_pipe
 *	Spawn an external program and connect a pipe to its stdin
 *
 * Args:
 *	cdp	 - Pointer to the cd_state_t structure
 *	progpath - Command line of program to spawn
 *	cpid	 - Child process ID return
 *	fmt	 - File header format to be written
 *	datalen	 - Audio data length in bytes
 *
 * Return:
 *	The pipe file descriptor, or -1 if failed.
 */
int
gen_open_pipe(cd_state_t *cdp, char *progpath, pid_t *cpid,
	      byte_t fmt, word32_t datalen)
{
	int	i,
		fd[2];

	if (!gen_set_eid(cdp))
		return FALSE;

	if (PIPE(fd) < 0) {
		(void) sprintf(cdp->i->msgbuf,
				"gen_open_pipe: output pipe failed (errno=%d)",
				errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
		(void) gen_reset_eid(cdp);
		return -1;
	}

	DBGPRN(DBG_DEVIO)(errfp, "gen_open_pipe: invoking [%s]\n", progpath);

	switch (*cpid = FORK()) {
	case -1:
		(void) sprintf(cdp->i->msgbuf,
			    "gen_open_pipe: fork failed (errno=%d)",
			    errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
		(void) gen_reset_eid(cdp);
		return FALSE;

	case 0:
		/* Child */

		/* Close unneeded pipe descriptors */
		(void) close(fd[1]);

		(void) close(0);	/* Close stdin */

		for (i = 3; i < 255; i++) {
			/* Close unneeded fds */
			if (i != fd[0])
				(void) close(i);
		}

		if (dup(fd[0]) < 0) {	/* Connect pipe to stdin */
			(void) fprintf(errfp,
				    "gen_open_pipe: dup failed (errno=%d)\n",
				    errno);
			exit(1);
		}

		(void) signal(SIGTERM, SIG_DFL);

		(void) execl(STR_SHPATH, STR_SHNAME, STR_SHARG,
			     progpath, NULL);
		(void) fprintf(errfp,
				"gen_open_pipe: exec failed (errno=%d)\n",
				errno);
		exit(255);
		/*NOTREACHED*/

	default:
		/* Parent */

		/* Close unneeded pipe descriptor */
		(void) close(fd[0]);
		break;
	}

	switch ((int) fmt) {
	case FILEFMT_AU:
		if (!gen_write_au_hdr(cdp, fd[1], datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	case FILEFMT_WAV:
		if (!gen_write_wav_hdr(cdp, fd[1], datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	case FILEFMT_AIFF:
		if (!gen_write_aiff_hdr(cdp, fd[1], datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	case FILEFMT_AIFC:
		if (!gen_write_aifc_hdr(cdp, fd[1], datalen)) {
			(void) gen_reset_eid(cdp);
			return -1;
		}
		break;

	default:
		break;
	}

	(void) gen_reset_eid(cdp);
	return (fd[1]);
}


/*
 * gen_close_pipe
 *	Close the program pipe that was opened via gen_open_pipe()
 *
 * Args:
 *	cdp - Pointer to the cd_state_t structure
 *	fd  - The pipe file descriptor
 *
 * Return:
 *	Nothing.
 */
void
gen_close_pipe(cd_state_t *cdp, int fd)
{
	if (!gen_set_eid(cdp))
		return;

	/* Close pipe */
	if (close(fd) < 0) {
		DBGPRN(DBG_DEVIO)(errfp,
			"gen_close_pipe: close failed (errno=%d)\n",
			errno);
		(void) gen_reset_eid(cdp);
		return;
	}

	(void) gen_reset_eid(cdp);
}


/*
 * gen_write_au_hdr
 *	Writes an au file header to the output file.
 *
 * Args:
 *	cdp	- Pointer to the cd_state_t structure
 *	fd	- Output file descriptor
 *	datalen	- Audio data length in bytes.  If this is 0, then the size
 *		  will be determined by checking the file.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
gen_write_au_hdr(cd_state_t *cdp, int fd, word32_t datalen)
{
	au_filehdr_t	hdr;

	if (!gen_set_eid(cdp))
		return FALSE;

	if (datalen == 0) {
		struct stat	stbuf;

		/* Get file information */
		if (fstat(fd, &stbuf) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_au_hdr: fstat failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}

		/* Set actual data length */
		datalen = stbuf.st_size - sizeof(hdr) - strlen(comment);

		/* Rewind to beginning of file */
		if (lseek(fd, (off_t) 0, SEEK_SET) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_au_hdr: lseek failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}
	}

	(void) memset(&hdr, 0, sizeof(hdr));

	/* Initialize an au file header. */
	hdr.au_magic = util_bswap32(AUDIO_AU_FILE_MAGIC);
	hdr.au_offset = util_bswap32(sizeof(hdr) + strlen(comment));
	hdr.au_data_size = util_bswap32(datalen);
	hdr.au_encoding = util_bswap32(AUDIO_AU_ENCODING_LINEAR_16);
	hdr.au_sample_rate = util_bswap32(44100);
	hdr.au_channels = util_bswap32(2);

	/* Write header */
	if (!gen_write_chunk(cdp, fd, "au header", (byte_t *) &hdr,
			     sizeof(hdr))) {
		(void) gen_reset_eid(cdp);
		return FALSE;
	}
	if (!gen_write_chunk(cdp, fd, "au comment", (byte_t *) comment,
			     strlen(comment))) {
		(void) gen_reset_eid(cdp);
		return FALSE;
	}

	(void) gen_reset_eid(cdp);
	return TRUE;
}


/*
 * gen_write_wav_hdr
 *	Writes an wav file header to the output file.
 *
 * Args:
 *	cdp	- Pointer to the cd_state_t structure
 *	fd	- Output file descriptor
 *	datalen	- Audio data length in bytes.  If this is 0, then the size
 *		  will be determined by checking the file.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
gen_write_wav_hdr(cd_state_t *cdp, int fd, word32_t datalen)
{
	wav_filehdr_t	hdr;

	if (!gen_set_eid(cdp))
		return FALSE;

	if (datalen == 0) {
		struct stat	stbuf;

		/* Get file information */
		if (fstat(fd, &stbuf) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_wav_hdr: fstat failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}

		/* Set actual data length */
		datalen = stbuf.st_size - sizeof(hdr);

		/* Rewind to beginning of file */
		if (lseek(fd, (off_t) 0, SEEK_SET) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_wav_hdr: lseek failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}
	}

	(void) memset(&hdr, 0, sizeof(hdr));

	/* RIFF chunk */
	(void) strncpy(hdr.r_riff, "RIFF", 4);
	hdr.r_length = util_lswap32(datalen + 36);
	(void) strncpy(hdr.r_wave, "WAVE", 4);

	/* FORMAT chunk */
	(void) strncpy(hdr.f_format, "fmt ", 4);
	hdr.f_length = util_lswap32(16);
	hdr.f_const = util_lswap16(1);
	hdr.f_channels = util_lswap16(2);
	hdr.f_sample_rate = util_lswap32(44100);
	hdr.f_bytes_per_s = util_lswap32(44100 * 2 * 2);
	hdr.f_bytes_per_sample = util_lswap16(2);
	hdr.f_bits_per_sample = util_lswap16(16);

	/* DATA chunk */
	(void) strncpy(hdr.d_data, "data", 4);
	hdr.d_length = util_lswap32(datalen);

	/* Write header */
	if (!gen_write_chunk(cdp, fd, "wav header", (byte_t *) &hdr,
			     sizeof(hdr))) {
		(void) gen_reset_eid(cdp);
		return FALSE;
	}

	(void) gen_reset_eid(cdp);
	return TRUE;
}


/*
 * gen_write_aiff_hdr
 *	Writes an aiff file header to the output file.
 *
 * Args:
 *	cdp	- Pointer to the cd_state_t structure
 *	fd	- Output file descriptor
 *	datalen	- Audio data length in bytes.  If this is 0, then the size
 *		  will be determined by checking the file.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
gen_write_aiff_hdr(cd_state_t *cdp, int fd, word32_t datalen)
{
	word32_t	tmp;
	aiff_filehdr_t	hdr;

	if (!gen_set_eid(cdp))
		return FALSE;

	if (datalen == 0) {
		struct stat	stbuf;

		/* Get file information */
		if (fstat(fd, &stbuf) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_aiff_hdr: fstat failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}

		/* Set actual data length */
		datalen = stbuf.st_size - AIFF_HDRSZ;

		/* Rewind to beginning of file */
		if (lseek(fd, (off_t) 0, SEEK_SET) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_aiff_hdr: lseek failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}
	}

	(void) memset(&hdr, 0, sizeof(hdr));

	/* FORM chunk */
	(void) strncpy(hdr.a_form, "FORM", 4);
	tmp = datalen + AIFF_HDRSZ - 8;
	hdr.a_length[0] = (tmp & 0xff000000) >> 24;
	hdr.a_length[1] = (tmp & 0x00ff0000) >> 16;
	hdr.a_length[2] = (tmp & 0x0000ff00) >> 8;
	hdr.a_length[3] = (tmp & 0x000000ff);
	(void) strncpy(hdr.a_aiff, "AIFF", 4);

	/* COMM chunk */
	(void) strncpy(hdr.c_comm, "COMM", 4);
	tmp = 18;
	hdr.c_length[0] = (tmp & 0xff000000) >> 24;
	hdr.c_length[1] = (tmp & 0x00ff0000) >> 16;
	hdr.c_length[2] = (tmp & 0x0000ff00) >> 8;
	hdr.c_length[3] = (tmp & 0x000000ff);
	tmp = 2;
	hdr.c_channels[0] = (tmp & 0xff00) >> 8;
	hdr.c_channels[1] = (tmp & 0x00ff);
	tmp = datalen >> 2;
	hdr.c_frames[0] = (tmp & 0xff000000) >> 24;
	hdr.c_frames[1] = (tmp & 0x00ff0000) >> 16;
	hdr.c_frames[2] = (tmp & 0x0000ff00) >> 8;
	hdr.c_frames[3] = (tmp & 0x000000ff);
	tmp = 16;
	hdr.c_sample_size[0] = (tmp & 0xff00) >> 8;
	hdr.c_sample_size[1] = (tmp & 0x00ff);
	(void) strncpy(hdr.c_sample_rate, "@\016\254D\0\0\0\0\0\0", 10);
				/* 44100 in 80-bit IEEE floating point */

	/* SSND chunk */
	(void) strncpy(hdr.s_ssnd, "SSND", 4);
	tmp = datalen + 8;
	hdr.s_length[0] = (tmp & 0xff000000) >> 24;
	hdr.s_length[1] = (tmp & 0x00ff0000) >> 16;
	hdr.s_length[2] = (tmp & 0x0000ff00) >> 8;
	hdr.s_length[3] = (tmp & 0x000000ff);

	/* Write header */
	if (!gen_write_chunk(cdp, fd, "aiff header", (byte_t *) &hdr,
			     AIFF_HDRSZ)) {
		(void) gen_reset_eid(cdp);
		return FALSE;
	}

	(void) gen_reset_eid(cdp);
	return TRUE;
}


/*
 * gen_write_aifc_hdr
 *	Writes an aifc file header to the output file.
 *
 * Args:
 *	cdp	- Pointer to the cd_state_t structure
 *	fd	- Output file descriptor
 *	datalen	- Audio data length in bytes.  If this is 0, then the size
 *		  will be determined by checking the file.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
gen_write_aifc_hdr(cd_state_t *cdp, int fd, word32_t datalen)
{
	word32_t	tmp;
	aifc_filehdr_t	hdr;

	if (!gen_set_eid(cdp))
		return FALSE;

	if (datalen == 0) {
		struct stat	stbuf;

		/* Get file information */
		if (fstat(fd, &stbuf) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_aifc_hdr: fstat failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}

		/* Set actual data length */
		datalen = stbuf.st_size - AIFC_HDRSZ;

		/* Rewind to beginning of file */
		if (lseek(fd, (off_t) 0, SEEK_SET) < 0) {
			(void) sprintf(cdp->i->msgbuf,
				"gen_write_aifc_hdr: lseek failed (errno=%d)",
				errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);
			(void) gen_reset_eid(cdp);
			return FALSE;
		}
	}

	(void) memset(&hdr, 0, sizeof(hdr));

	/* FORM chunk */
	(void) strncpy(hdr.a_form, "FORM", 4);
	tmp = datalen + AIFC_HDRSZ - 8;
	hdr.a_length[0] = (tmp & 0xff000000) >> 24;
	hdr.a_length[1] = (tmp & 0x00ff0000) >> 16;
	hdr.a_length[2] = (tmp & 0x0000ff00) >> 8;
	hdr.a_length[3] = (tmp & 0x000000ff);
	(void) strncpy(hdr.a_aifc, "AIFC", 4);

	/* FVER chunk */
	(void) strncpy(hdr.f_fver, "FVER", 4);
	tmp = 4;
	hdr.f_length[0] = (tmp & 0xff000000) >> 24;
	hdr.f_length[1] = (tmp & 0x00ff0000) >> 16;
	hdr.f_length[2] = (tmp & 0x0000ff00) >> 8;
	hdr.f_length[3] = (tmp & 0x000000ff);
	tmp = 2726318400UL;
	hdr.f_version[0] = (tmp & 0xff000000) >> 24;
	hdr.f_version[1] = (tmp & 0x00ff0000) >> 16;
	hdr.f_version[2] = (tmp & 0x0000ff00) >> 8;
	hdr.f_version[3] = (tmp & 0x000000ff);

	/* COMM chunk */
	(void) strncpy(hdr.c_comm, "COMM", 4);
	tmp = 18;
	hdr.c_length[0] = (tmp & 0xff000000) >> 24;
	hdr.c_length[1] = (tmp & 0x00ff0000) >> 16;
	hdr.c_length[2] = (tmp & 0x0000ff00) >> 8;
	hdr.c_length[3] = (tmp & 0x000000ff);
	tmp = 2;
	hdr.c_channels[0] = (tmp & 0xff00) >> 8;
	hdr.c_channels[1] = (tmp & 0x00ff);
	tmp = datalen >> 2;
	hdr.c_frames[0] = (tmp & 0xff000000) >> 24;
	hdr.c_frames[1] = (tmp & 0x00ff0000) >> 16;
	hdr.c_frames[2] = (tmp & 0x0000ff00) >> 8;
	hdr.c_frames[3] = (tmp & 0x000000ff);
	tmp = 16;
	hdr.c_sample_size[0] = (tmp & 0xff00) >> 8;
	hdr.c_sample_size[1] = (tmp & 0x00ff);
	(void) strncpy(hdr.c_sample_rate, "@\016\254D\0\0\0\0\0\0", 10);
				/* 44100 in 80-bit IEEE floating point */
	(void) strncpy(hdr.c_comptype, "NONE", 4);
	hdr.c_complength = 14;
	(void) strncpy(hdr.c_compstr, "not compressed", 14);

	/* SSND chunk */
	(void) strncpy(hdr.s_ssnd, "SSND", 4);
	tmp = datalen + 8;
	hdr.s_length[0] = (tmp & 0xff000000) >> 24;
	hdr.s_length[1] = (tmp & 0x00ff0000) >> 16;
	hdr.s_length[2] = (tmp & 0x0000ff00) >> 8;
	hdr.s_length[3] = (tmp & 0x000000ff);

	/* Write header */
	if (!gen_write_chunk(cdp, fd, "aifc header", (byte_t *) &hdr,
			     AIFC_HDRSZ)) {
		(void) gen_reset_eid(cdp);
		return FALSE;
	}

	(void) gen_reset_eid(cdp);
	return TRUE;
}


/*
 * gen_write_chunk
 *	Writes data to the fd, catering for possible interrupts.
 *
 * Args:
 *	cdp	- Pointer to the cd_state_t structure
 *	fd	- File descriptor to device or file
 *	path	- Path name to device or file
 *	data	- Data to write
 *	len	- Data length
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
gen_write_chunk(cd_state_t *cdp, int fd, char *path, byte_t *data, size_t len)
{
	int	offset = 0,
		cnt = 0,
		n;

	if (!gen_set_eid(cdp))
		return FALSE;

	DBGPRN(DBG_DEVIO)(errfp, "\nWrite [%s]: %d bytes\n", path, len);

	/* Write out */
	do {
		errno = 0;
		n = write(fd, &data[offset], len - offset);

		/* Have we finished? */
		if (n == (len - offset)) {
			(void) gen_reset_eid(cdp);
			return TRUE;
		}

		/* Cater for EINTR */
		if (n < 0) {
			if (errno == EINTR) {
				/* Reset and increment counter if not paused */
				n = 0;
				if (cdp->i->state != CDSTAT_PAUSED)
					cnt++;
			}
			else
				break;
		}

		offset += n;

	} while (offset < len && cnt < CDDA_INTR_MAX);

	(void) sprintf(cdp->i->msgbuf,
			"gen_write_chunk: write failed (errno=%d)",
			errno);
	DBGPRN(DBG_DEVIO)(errfp, "%s\n", cdp->i->msgbuf);

	(void) gen_reset_eid(cdp);
	return FALSE;
}


/*
 * gen_byteswap
 *	Carry out byte swapping.
 *
 * Args:
 *	srcbuf - audio data source buffer
 *	tgtbuf - audio data target buffer
 *	len    - data length in bytes
 *
 * Return:
 *	Nothing.
 */
void
gen_byteswap(byte_t *srcbuf, byte_t *tgtbuf, size_t len)
{
	int	i;

	/* Run through samples */
	for (i = 0; i < (int) len; i += 2) {
		/* Byte swapping */
		tgtbuf[i] = srcbuf[i+1];
		tgtbuf[i+1] = srcbuf[i];;
	}
}


/*
 * gen_chroute
 *	Other optional processing may need be carried out on the audio
 *	prior to sending it to the audio file.
 *
 *	CHROUTE_NORMAL		Leave as is
 *	CHROUTE_REVERSE		Swaps left and right
 *	CHROUTE_L_MONO		Feed left to both channels
 *	CHROUTE_R_MONO	        Feed right to both channels
 *	CHROUTE_MONO		Feed left and right avrg to both channels
 *
 * Args:
 *	data - Data to transform
 *	len  - Data length in bytes
 *
 * Return:
 *	Nothing.
 */
void
gen_chroute(int chroute, sword16_t *data, size_t len)
{
	int		l,
			r;
	sword16_t	temp;

	switch (chroute) {
	case CHROUTE_NORMAL:
		break;

	case CHROUTE_REVERSE:
		for (l = 0; l < len; l += sizeof(sword16_t)) {
			/* Swap left and right samples */
			r = l + 1;
			temp = *(&data[r]);
			*(&data[r]) = *(&data[l]);
			*(&data[l]) = temp;
		}
		break;

	case CHROUTE_L_MONO:
		for (l = 0; l < len; l += sizeof(sword16_t)) {
			/* Make right same as left */
			r = l + 1;
			*(&data[r]) = *(&data[l]);
		}
		break;

	case CHROUTE_R_MONO:
		for (l = 0; l < len; l += sizeof(sword16_t)) {
			/* Make left same as right */
			r = l + 1;
			*(&data[l]) = *(&data[r]);
		}
		break;

	case CHROUTE_MONO:
		for (l = 0; l < len; l += sizeof(sword16_t)) {
			/* Average left and right */
			r = l + 1;
			temp = (*(&data[l]) + *(&data[r])) >> 1;
			*(&data[l]) = temp;
			*(&data[r]) = temp;
		}
		break;

	default:
		break;
	}
}


/*
 * gen_write_init
 *	Generic write services initialization
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
void
gen_write_init(void)
{
	static bool_t	gen_write_initted = FALSE;

	if (!gen_write_initted) {
		gen_write_initted = TRUE;

		ruid = util_get_ouid();
		rgid = util_get_ogid();
		euid = geteuid();
		egid = getegid();
	}
}

#endif	/* CDDA_SYSVIPC */


