/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   Linux CDDA read ioctl support
 */
#ifndef LINT
static char *_lxread_c_ident_ = "@(#)lxread.c	7.20 02/04/09";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_RD_LX) && defined(CDDA_SYSVIPC)

#include <linux/cdrom.h>
#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/lxread.h"

extern appdata_t	app_data;
extern FILE		*errfp;
extern void		*cdda_shmaddr;

STATIC int		lx_rsemid;	/* Semaphores identifier */
STATIC cd_state_t	*lx_rcd;	/* CD state pointer */


/*
 * lxcd_read_fillbuf
 *	Before beginning to play we fill the data buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
lxcd_read_fillbuf(int fd)
{
	int			i;
	byte_t			*start,
				*end,
				*p;
	struct cdrom_read_audio	cdda;

	/* Get lock */
	cdda_waitsem(lx_rsemid, LOCK);

	/* Initialize */
	start = &lx_rcd->cdb->data[lx_rcd->cds->olap_bytes];

	/* Set up for read */
	cdda.addr.lba = lx_rcd->i->start_lba;
	cdda.addr_format = CDROM_LBA;
	cdda.nframes = lx_rcd->cds->chunk_blocks + lx_rcd->cds->olap_blocks;
	cdda.buf = (u_char *) start;

	cdda_rd_setcurtrk(lx_rcd, (int) cdda.addr.lba);

	/* Fill up our buffer */
	i = 0;

	while (i < lx_rcd->cds->buffer_chunks && !lx_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * lx_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (cdda.addr.lba + lx_rcd->cds->chunk_blocks >=
		    lx_rcd->i->end_lba) {
			cdda.nframes = lx_rcd->i->end_lba - cdda.addr.lba;
			(void) memset(
				lx_rcd->cdb->data,
				0,
				lx_rcd->cds->chunk_bytes +
				(lx_rcd->cds->olap_bytes << 1)
			);
			lx_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDROMREADAUDIO, &cdda) < 0) {
			(void) sprintf(lx_rcd->i->msgbuf,
					"lxcd_read_fillbuf: CDROMREADAUDIO "
					"ioctl failed: %s",
					strerror(errno));
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", lx_rcd->i->msgbuf);
			cdda_postsem(lx_rsemid, LOCK);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
				"\nSent CDROMREADAUDIO ioctl: blk=%d len=%d\n",
				cdda.addr.lba, cdda.nframes);

		/* Do jitter correction */
		if (i == 0)
			p = &lx_rcd->cdb->data[lx_rcd->cds->olap_bytes];
		else
			p = cdda_rd_corrjitter(lx_rcd);

		/* Data end */
		end = p + lx_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (lx_rcd->i->jitter) {
			(void) memcpy(
				lx_rcd->cdb->olap,
				end - lx_rcd->cds->search_bytes,
				lx_rcd->cds->search_bytes << 1
			);
		}

		/* Copy in */
		(void) memcpy(
			&lx_rcd->cdb->b[
			    lx_rcd->cds->chunk_bytes * lx_rcd->cdb->nextin
			],
			p,
			lx_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		lx_rcd->cdb->nextin++;
		lx_rcd->cdb->nextin %= lx_rcd->cds->buffer_chunks;
		lx_rcd->cdb->occupied++;

		cdda.addr.lba += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.addr.lba++;

		cdda_rd_setcurtrk(lx_rcd, (int) cdda.addr.lba);

		i++;

		/* Update debug level */
		app_data.debug = lx_rcd->i->debug;
	}

	/* No room available now */
	cdda_waitsem(lx_rsemid, ROOM);

	/* Signal data available */
	cdda_postsem(lx_rsemid, DATA);

	/* Release lock */
	cdda_postsem(lx_rsemid, LOCK);

	return TRUE;
}


/*
 * lxcd_read_cont
 *	Function responsible for reading from the cd,
 *	correcting for jitter and writing to the buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
lxcd_read_cont(int fd)
{
	byte_t			*start,
				*end,
				*p;
	struct cdrom_read_audio	cdda;

	/* Initialize */
	start = &lx_rcd->cdb->data[lx_rcd->cds->olap_bytes];

	/* Set up for read */
	cdda.addr.lba = lx_rcd->i->start_lba + lx_rcd->cds->buffer_blocks;
	cdda.addr_format = CDROM_LBA;
	cdda.nframes = lx_rcd->cds->chunk_blocks + lx_rcd->cds->olap_blocks;
	cdda.buf = (u_char *) start;

	cdda_rd_setcurtrk(lx_rcd, (int) cdda.addr.lba);

	/* While not stopped or finished */
	while (lx_rcd->i->state != CDSTAT_COMPLETED && !lx_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * lx_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (cdda.addr.lba + lx_rcd->cds->chunk_blocks >=
		    lx_rcd->i->end_lba) {
			cdda.nframes = lx_rcd->i->end_lba - cdda.addr.lba;
			(void) memset(
				lx_rcd->cdb->data,
				0,
				lx_rcd->cds->chunk_bytes +
				(lx_rcd->cds->olap_bytes << 1)
			);
			lx_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDROMREADAUDIO, &cdda) < 0) {
			(void) sprintf(lx_rcd->i->msgbuf,
					"lxcd_read_cont: CDROMREADAUDIO ioctl "
					"failed: %s",
					strerror(errno));
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", lx_rcd->i->msgbuf);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
				"\nSent CDROMREADAUDIO ioctl: blk=%d len=%d\n",
				cdda.addr.lba, cdda.nframes);

		/* Do jitter correction */
		p = cdda_rd_corrjitter(lx_rcd);

		/* Data end */
		end = p + lx_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (lx_rcd->i->jitter) {
			(void) memcpy(
				lx_rcd->cdb->olap,
				end - lx_rcd->cds->search_bytes,
				lx_rcd->cds->search_bytes << 1
			);
		}

		/* Get lock */
		cdda_waitsem(lx_rsemid, LOCK);

		/* Wait until there is room */
		while (lx_rcd->cdb->occupied >= lx_rcd->cds->buffer_chunks &&
		       lx_rcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(lx_rsemid, LOCK);
			cdda_waitsem(lx_rsemid, ROOM);
			cdda_waitsem(lx_rsemid, LOCK);
		}

		/* Break if completed */
		if (lx_rcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(lx_rsemid, LOCK);
			break;
		}

		/* Copy to free location */
		(void) memcpy(
			&lx_rcd->cdb->b[
			    lx_rcd->cds->chunk_bytes * lx_rcd->cdb->nextin
			],
			p,
			lx_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		lx_rcd->cdb->nextin++;
		lx_rcd->cdb->nextin %= lx_rcd->cds->buffer_chunks;
		lx_rcd->cdb->occupied++;

		cdda.addr.lba += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.addr.lba++;

		cdda_rd_setcurtrk(lx_rcd, (int) cdda.addr.lba);

		/* Signal data available */
		cdda_postsem(lx_rsemid, DATA);

		/* Update debug level */
		app_data.debug = lx_rcd->i->debug;

		/* Release lock */
		cdda_postsem(lx_rsemid, LOCK);
	}

	/* Wake up writer */
	cdda_postsem(lx_rsemid, DATA);

	return TRUE;
}


/*
 * lxcd_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killwriter - whether to kill the writer process
 *
 * Return:
 *	Nothing.
 */
STATIC void
lxcd_cleanup(bool_t killwriter)
{
	DBGPRN(DBG_DEVIO)(errfp,
			  "\nlxcd_cleanup: Cleaning up reader pid=%d\n",
			  (int) getpid());

	(void) sleep(1);

	if (lx_rcd != NULL) {
		if (killwriter && lx_rcd->i != NULL && lx_rcd->i->writer > 0) {
			(void) kill(lx_rcd->i->writer, SIGTERM);
			lx_rcd->i->writer = 0;
			lx_rcd->i->state = CDSTAT_COMPLETED;
		}

		MEM_FREE(lx_rcd);
		lx_rcd = NULL;
	}
}


/*
 * lxcd_onterm
 *	SIGTERM signal handler - gracefully exit
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
lxcd_onterm(int signo)
{
	lxcd_cleanup(FALSE);
	exit(1);
}


/*
 * lx_readinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
lx_readinit(void)
{
	return CDDA_READAUDIO;
}


/*
 * lx_read
 *	Attaches to shared memory and semaphores. Continuously reads
 *	data from the cd and places into shared memory.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
lx_read(int fd)
{
	struct sigaction	action_term;

	/* Check configuration */
	if (app_data.di_method != 1 /* DI_SLIOC */) {
		(void) fprintf(errfp,
			"lx_read: Inconsistent deviceInterfaceMethod "
			"and cddaReadMethod parameters.  Aborting.\n");
		return FALSE;
	}

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);

	action_term.sa_handler = lxcd_onterm;
	sigemptyset(&action_term.sa_mask);
	action_term.sa_flags = 0;
	if (sigaction(SIGTERM, &action_term, NULL) < 0) {
		(void) fprintf(errfp,
				"lx_read: sigaction failed (SIGTERM): %s\n",
				strerror(errno));
		return FALSE;
	}

	/* Allocate mempry */
	lx_rcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (lx_rcd == NULL) {
		(void) fprintf(errfp, "lx_read: out of memory\n");
		return FALSE;
	}

	(void) memset(lx_rcd, 0, sizeof(cd_state_t));

	/* Initialize cd pointers to point into shared memory */
	if ((lx_rsemid = cdda_rw_initipc(lx_rcd)) < 0) {
		lxcd_cleanup(TRUE);
		return FALSE;
	}

	DBGPRN(DBG_DEVIO)(errfp,
			  "\nlxcd_read: Reading CDDA: "
			  "chunk_blks=%d, olap_blks=%d\n",
			  lx_rcd->cds->chunk_blocks, lx_rcd->cds->olap_blocks);

	/* Fill data buffer */
	if (!lxcd_read_fillbuf(fd)) {
		lxcd_cleanup(TRUE);
		return FALSE;
	}

	/* Keep reading */
	if (!lxcd_read_cont(fd)) {
		lxcd_cleanup(TRUE);
		return FALSE;
	}

	lxcd_cleanup(FALSE);
	return TRUE;
}

#endif	/* CDDA_RD_LX CDDA_SYSVIPC */

