/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   OSS (Open Sound System) and Linux sound driver support
 */
#ifndef LINT
static char *_osswrite_c_ident_ = "@(#)osswrite.c	7.47 02/04/23";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_WR_OSS) && defined(CDDA_SYSVIPC)

#ifdef USE_SYS_SOUNDCARD_H
#include <sys/soundcard.h>
#else
#include "cdda_d/soundcard.h"
#endif
#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/genwrite.h"
#include "cdda_d/osswrite.h"


extern appdata_t	app_data;
extern FILE		*errfp;

STATIC int		oss_audio_fd = -1,	/* Audio device fd */
			oss_file_fd = -1,	/* Output file fd */
			oss_pipe_fd = -1,	/* Pipe to program fd */
			oss_wsemid;		/* Semaphores identifier */
STATIC pid_t		oss_pipe_pid = -1;	/* Pipe to program pid */
STATIC cd_state_t	*oss_wcd;		/* Shared memory pointer */
STATIC char		*oss_device,		/* Audio device */
			*oss_pipeprog;		/* Pipe to program */
STATIC bool_t		oss_write_dev = FALSE,	/* Write to audio device */
			oss_write_file = FALSE,	/* Write to output file */
			oss_write_pipe = FALSE,	/* Pipe to program */
			oss_mixer_supp = FALSE,	/* OSS PCM mixer supported */
			oss_file_be = FALSE;	/* Big endian output file */
STATIC byte_t		oss_file_fmt;		/* Output file format */
STATIC ossaudio_info_t	oss_prev_status;	/* Audio hardware settings */


/*
 * ossaud_get_info
 *	Get audio hardware settings.
 *
 * Args:
 *	info - Info to return
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
ossaud_get_info(ossaudio_info_t *info)
{
	int	val;

	/* Retrieve settings */

	/* Sample format */
	val = AFMT_QUERY;
	if (ioctl(oss_audio_fd, SNDCTL_DSP_SETFMT, &val) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_get_info: SNDCTL_DSP_SETFMT failed "
			    "(errno=%d)",
			    errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	info->fmt = val;

	/* Stereo/mono */
	val = -1;
	if (ioctl(oss_audio_fd, SOUND_PCM_READ_CHANNELS, &val) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_get_info: SOUND_PCM_READ_CHANNELS failed "
			    "(errno=%d)",
			    errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	info->nch = val;

	/* Sampling rate */
	val = -1;
	if (ioctl(oss_audio_fd, SOUND_PCM_READ_BITS, &val) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_get_info: SOUND_PCM_READ_BITS failed "
			    "(errno=%d)",
			    errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	info->samp = val;

	/* PCM mixer setting */
	if (ioctl(oss_audio_fd, SOUND_MIXER_READ_DEVMASK, &val) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_get_info: SOUND_MIXER_READ_DEVMASK "
			    "failed (errno=%d)",
			    errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
	}
	else if ((val & SOUND_MASK_PCM) != 0) {
		oss_mixer_supp = TRUE;
	}

	if (oss_mixer_supp) {
		if (ioctl(oss_audio_fd, SOUND_MIXER_READ_PCM, &val) < 0) {
			(void) sprintf(oss_wcd->i->msgbuf,
				    "ossaud_get_info: SOUND_MIXER_READ_PCM "
				    "failed (errno=%d)",
				    errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
			info->vol0 = 100;
			info->vol1 = 100;
		}
		else {
			info->vol0 = (int) (val & 0xff);
			info->vol1 = (int) ((val >> 8) & 0xff);
		}
	}

	return TRUE;
}


/*
 * ossaud_set_info
 *	Set audio hardware settings.
 *
 * Args:
 *	info - Info to set
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
ossaud_set_info(ossaudio_info_t *info)
{
	int	val;

	/* Apply settings */

	/* Sample format */
	val = info->fmt;
	if (ioctl(oss_audio_fd, SNDCTL_DSP_SETFMT, &val) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_set_info: SNDCTL_DSP_SETFMT failed "
			    "(errno=%d)",
			    errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	if (val != info->fmt) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_set_info: Audio hardware does not "
			    "support the required sample format.");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	info->fmt = val;

	/* Stereo/mono */
	val = info->nch;
	if (ioctl(oss_audio_fd, SNDCTL_DSP_CHANNELS, &val) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_set_info: SNDCTL_DSP_CHANNELS failed "
			    "(errno=%d)",
			    errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	if (val != info->nch) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_set_info: Audio hardware does not "
			    "support %d-channel mode.",
			    info->nch);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	info->nch = val;

	/* Sampling rate */
	val = info->samp;
	if (ioctl(oss_audio_fd, SNDCTL_DSP_SPEED, &val) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_set_info: SNDCTL_DSP_SPEED failed "
			    "(errno=%d)",
			    errno);
		return FALSE;
	}
	if (abs(info->samp - val) > (info->samp / 20)) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_set_info: Audio hardware does not "
			    "support %dHz sampling rate.",
			    info->samp);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}
	info->samp = val;

	if (oss_mixer_supp) {
		/* PCM mixer setting */
		val = ((info->vol1 & 0xff) << 8) | (info->vol0 & 0xff);
		if (ioctl(oss_audio_fd, SOUND_MIXER_WRITE_PCM, &val) < 0) {
			(void) sprintf(oss_wcd->i->msgbuf,
				    "ossaud_set_info: SOUND_MIXER_WRITE_PCM "
				    "failed (errno=%d)",
				    errno);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		}
	}

	return TRUE;
}


/*
 * ossaud_open_dev
 *	Opens the audio device.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
ossaud_open_dev(void)
{
	char		*cp;
	struct stat	stbuf;

	if (!gen_set_eid(oss_wcd))
		return FALSE;

	if (oss_device != NULL) {
		MEM_FREE(oss_device);
		oss_device = NULL;
	}

	/* Use environment variable if set */
	if ((cp = getenv("AUDIODEV")) == NULL)
		cp = DEFAULT_DEV_DSP;

	if (!util_newstr(&oss_device, cp)) {
		(void) sprintf(oss_wcd->i->msgbuf,
				"ossaud_open_dev: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		(void) gen_reset_eid(oss_wcd);
		return FALSE;
	}

	/* Check to make sure it's a device */
	if (stat(oss_device, &stbuf) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_open_dev: stat of %s failed (errno=%d)",
			    oss_device, errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		(void) gen_reset_eid(oss_wcd);
		return FALSE;
	}
	if (!S_ISCHR(stbuf.st_mode)) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_open_dev: %s is not a character device",
			    oss_device);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		(void) gen_reset_eid(oss_wcd);
		return FALSE;
	}

	/* Open audio device */
	if ((oss_audio_fd = open(oss_device, O_WRONLY, 0)) < 0) {
		(void) sprintf(oss_wcd->i->msgbuf,
			    "ossaud_open_dev: open of %s failed (errno=%d)",
			    oss_device, errno);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		(void) gen_reset_eid(oss_wcd);
		return FALSE;
	}

	(void) gen_reset_eid(oss_wcd);
	return TRUE;
}


/*
 * ossaud_close_dev
 *	Closes the audio device.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
ossaud_close_dev(void)
{
	if (!gen_set_eid(oss_wcd))
		return FALSE;

	/* Close audio device */
	if (close(oss_audio_fd) < 0) {
		DBGPRN(DBG_DEVIO)(errfp,
			"ossaud_close_dev: close failed (errno=%d)\n",
			errno);
		(void) gen_reset_eid(oss_wcd);
		return FALSE;
	}

	oss_audio_fd = -1;

	(void) gen_reset_eid(oss_wcd);
	return TRUE;
}


/*
 * ossaud_config
 *	Sets up audio device for playing CD quality audio.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
ossaud_config(void)
{
	ossaudio_info_t	ossaud_info;
	int		temp;

	/* Play side */
	ossaud_info.samp = 44100;
	ossaud_info.nch = 2;
	ossaud_info.fmt = AFMT_S16_LE;

	/* Apply new settings */
	if (!ossaud_set_info(&ossaud_info))
		return FALSE;

	/* Retrieve current settings */
	if (!ossaud_get_info(&ossaud_info))
		return FALSE;

	oss_prev_status = ossaud_info;	/* structure copy */

	/* Update volume and balance */
	if (ossaud_info.vol0 == ossaud_info.vol1) {
		oss_wcd->i->vol = util_untaper_vol(ossaud_info.vol0);
		oss_wcd->i->vol_left = 100;
		oss_wcd->i->vol_right = 100;
	}
	else if (ossaud_info.vol0 > ossaud_info.vol1) {
		oss_wcd->i->vol = util_untaper_vol(ossaud_info.vol0);
		oss_wcd->i->vol_left = 100;
		oss_wcd->i->vol_right =
			(ossaud_info.vol1 * 100) / ossaud_info.vol0;
		temp = ossaud_info.vol0 >> 1;
		if (((ossaud_info.vol1 * 100) % ossaud_info.vol0) >= temp)
			oss_wcd->i->vol_right++;
	}
	else {
		oss_wcd->i->vol = util_untaper_vol(ossaud_info.vol1);
		oss_wcd->i->vol_right = 100;
		oss_wcd->i->vol_left =
			(ossaud_info.vol0 * 100) / ossaud_info.vol1;
		temp = ossaud_info.vol1 >> 1;
		if (((ossaud_info.vol0 * 100) % ossaud_info.vol1) >= temp)
			oss_wcd->i->vol_left++;
	}

	return TRUE;
}


/*
 * ossaud_update
 *	Updates hardware volume and balance settings from GUI settings.
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
STATIC void
ossaud_update(void)
{
	ossaudio_info_t		ossaud_info;
	int			temp;
	static int		prev_vol0 = -1,
				prev_vol1 = -1;

	app_data.vol_taper = oss_wcd->i->vol_taper;

	/* Update volume and balance */
	if (oss_wcd->i->vol_left == oss_wcd->i->vol_right) {
		ossaud_info.vol0 = util_taper_vol(oss_wcd->i->vol);
		ossaud_info.vol1 = ossaud_info.vol0;
	}
	else if (oss_wcd->i->vol_left > oss_wcd->i->vol_right) {
		temp = (oss_wcd->i->vol * oss_wcd->i->vol_right) / 100;
		if (((oss_wcd->i->vol * oss_wcd->i->vol_right) % 100) >= 50)
			temp++;
		ossaud_info.vol0 = util_taper_vol(oss_wcd->i->vol);
		ossaud_info.vol1 = util_taper_vol(temp);
	}
	else {
		temp = (oss_wcd->i->vol * oss_wcd->i->vol_left) / 100;
		if (((oss_wcd->i->vol * oss_wcd->i->vol_left) % 100) >= 50)
			temp++;
		ossaud_info.vol0 = util_taper_vol(temp);
		ossaud_info.vol1 = util_taper_vol(oss_wcd->i->vol);
	}

	if (ossaud_info.vol0 == prev_vol0 && ossaud_info.vol1 == prev_vol1)
		/* No change */
		return;

	ossaud_info.samp = 44100;
	ossaud_info.nch = 2;
	ossaud_info.fmt = AFMT_S16_LE;

	prev_vol0 = ossaud_info.vol0;
	prev_vol1 = ossaud_info.vol1;

	/* Apply new settings */
	(void) ossaud_set_info(&ossaud_info);
}


/*
 * ossaud_status
 *	Updates volume and balance settings from audio device.
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
STATIC void
ossaud_status(void)
{
	ossaudio_info_t	ossaud_info;
	int		temp;

	/* Retrieve current settings */
	if (!ossaud_get_info(&ossaud_info))
		return;

	if (memcmp((byte_t *) &ossaud_info, (byte_t *) &oss_prev_status,
	    sizeof(ossaud_info)) == 0)
		/* No change */
		return;

	oss_prev_status = ossaud_info;	/* structure copy */

	/* Update volume and balance */
	if (ossaud_info.vol0 == ossaud_info.vol1) {
		oss_wcd->i->vol = util_untaper_vol(ossaud_info.vol0);
		oss_wcd->i->vol_left = 100;
		oss_wcd->i->vol_right = 100;
	}
	else if (ossaud_info.vol0 > ossaud_info.vol1) {
		oss_wcd->i->vol = util_untaper_vol(ossaud_info.vol0);
		oss_wcd->i->vol_left = 100;
		oss_wcd->i->vol_right =
			(ossaud_info.vol1 * 100) / ossaud_info.vol0;
		temp = ossaud_info.vol0 >> 1;
		if (((ossaud_info.vol1 * 100) % ossaud_info.vol0) >= temp)
			oss_wcd->i->vol_right++;
	}
	else {
		oss_wcd->i->vol = util_untaper_vol(ossaud_info.vol1);
		oss_wcd->i->vol_right = 100;
		oss_wcd->i->vol_left =
			(ossaud_info.vol0 * 100) / ossaud_info.vol1;
		temp = ossaud_info.vol1 >> 1;
		if (((ossaud_info.vol0 * 100) % ossaud_info.vol1) >= temp)
			oss_wcd->i->vol_left++;
	}
}


/*
 * ossaud_write
 *	Function respsonsible for writing from the buffer to the audio
 *	device (and file if saving).
 *
 * Args:
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
ossaud_write(curstat_t *s)
{
	byte_t	*in_data,
		*sw_data,
		*le_data,
		*be_data,
		*dev_data,
		*file_data,
		*pipe_data;
	char	*fname = NULL;
	int	trk_idx = -1;
	bool_t	ret;

	/* Get memory for audio device write buffer */
	le_data = (byte_t *) MEM_ALLOC("le_data", oss_wcd->cds->chunk_bytes);
	if (le_data == NULL) {
		(void) sprintf(oss_wcd->i->msgbuf,
				"ossaud_write: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}

	/* Get memory for audio file write buffer */
	be_data = (byte_t *) MEM_ALLOC("be_data", oss_wcd->cds->chunk_bytes);
	if (be_data == NULL) {
		(void) sprintf(oss_wcd->i->msgbuf,
				"ossaud_write: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", oss_wcd->i->msgbuf);
		return FALSE;
	}

	(void) memset(le_data, 0, oss_wcd->cds->chunk_bytes);
	(void) memset(be_data, 0, oss_wcd->cds->chunk_bytes);

	in_data = app_data.cdda_bigendian ? be_data : le_data;
	sw_data = app_data.cdda_bigendian ? le_data : be_data;

	/* While not stopped */
	ret = TRUE;
	while (oss_wcd->i->state != CDSTAT_COMPLETED) {
		/* Get lock */
		cdda_waitsem(oss_wsemid, LOCK);

		/* End if finished and nothing in the buffer */
		if (oss_wcd->cdb->occupied == 0 && oss_wcd->i->cdda_done) {
			oss_wcd->i->state = CDSTAT_COMPLETED;
			cdda_postsem(oss_wsemid, LOCK);
			break;
		}

		/* Wait until there is something there */
		while (oss_wcd->cdb->occupied <= 0 &&
		       oss_wcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(oss_wsemid, LOCK);
			cdda_waitsem(oss_wsemid, DATA);
			cdda_waitsem(oss_wsemid, LOCK);
		}

		/* Break if stopped */
		if (oss_wcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(oss_wsemid, LOCK);
			break;
		}

		/* Copy a chunk from the nextout slot into the data buffers */
		(void) memcpy(
			in_data,
			&oss_wcd->cdb->b[
			    oss_wcd->cds->chunk_bytes * oss_wcd->cdb->nextout
			],
			oss_wcd->cds->chunk_bytes
		);

		/* Update pointers */
		oss_wcd->cdb->nextout++;
		oss_wcd->cdb->nextout %= oss_wcd->cds->buffer_chunks;
		oss_wcd->cdb->occupied--;

		/* Signal room available */
		cdda_postsem(oss_wsemid, ROOM);

		/* Release lock */
		cdda_postsem(oss_wsemid, LOCK);

		/* Change channel routing if necessary */
		gen_chroute(oss_wcd->i->chroute, (sword16_t *)(void *) in_data,
			    (size_t) oss_wcd->cds->chunk_bytes);

		/* Prepare byte-swapped version of the data */
		gen_byteswap(in_data, sw_data,
			     (size_t) oss_wcd->cds->chunk_bytes);

		/* Write to device */
		if (oss_write_dev) {
			/* Always feed OSS little endian (AFMT_16_LE) data */
			dev_data = le_data;

			if (!gen_write_chunk(oss_wcd, oss_audio_fd,
					oss_device, dev_data,
					(size_t) oss_wcd->cds->chunk_bytes)) {
				oss_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(oss_wsemid, LOCK);
				ret = FALSE;
				break;
			}

			/* Update volume and balance settings */
			ossaud_update();

			if ((oss_wcd->i->frm_played % FRAME_PER_SEC) == 0)
				ossaud_status();
		}

		/* Write to file */
		if (oss_write_file) {
			file_data = oss_file_be ? be_data : le_data;

			/* Open new track file if necessary */
			if (!app_data.cdda_trkfile) {
				fname = s->trkinfo[0].outfile;
			}
			else if (trk_idx != oss_wcd->i->trk_idx) {
				trk_idx = oss_wcd->i->trk_idx;

				if (oss_file_fd >= 0)
					gen_close_file(
						oss_wcd,
						oss_file_fd,
						oss_file_fmt
					);

				fname = s->trkinfo[trk_idx].outfile;
				oss_file_fd = gen_open_file(
					oss_wcd,
					fname,
					O_WRONLY | O_TRUNC | O_CREAT,
					S_IRUSR | S_IWUSR,
					oss_file_fmt,
					oss_wcd->i->trk_len
				);
				if (oss_file_fd < 0) {
					oss_wcd->i->state = CDSTAT_COMPLETED;
					cdda_postsem(oss_wsemid, LOCK);
					ret = FALSE;
					break;
				}

				DBGPRN(DBG_DEVIO)(errfp,
					  "\nflaud_write: Writing file [%s]: "
					  "chunk_blks=%d\n",
					  fname, oss_wcd->cds->chunk_blocks);
			}

			if (!gen_write_chunk(oss_wcd, oss_file_fd,
					fname, file_data,
					(size_t) oss_wcd->cds->chunk_bytes)) {
				oss_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(oss_wsemid, LOCK);
				ret = FALSE;
				break;
			}
		}

		/* Pipe to program */
		if (oss_write_pipe) {
			pipe_data = oss_file_be ? be_data : le_data;

			if (!gen_write_chunk(oss_wcd, oss_pipe_fd,
					oss_pipeprog, pipe_data,
					(size_t) oss_wcd->cds->chunk_bytes)) {
				oss_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(oss_wsemid, LOCK);
				ret = FALSE;
				break;
			}
		}

		oss_wcd->i->frm_played += oss_wcd->cds->chunk_blocks;

		while (oss_wcd->i->state == CDSTAT_PAUSED)
			util_delayms(400);

		/* Update debug level */
		app_data.debug = oss_wcd->i->debug;
	}

	/* Wake up reader */
	cdda_postsem(oss_wsemid, ROOM);

	/* Free memory */
	MEM_FREE(le_data);
	MEM_FREE(be_data);

	return (ret);
}


/*
 * ossaud_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killreader - Whether to kill the reader process
 *
 * Return:
 *	Nothing.
 */
STATIC void
ossaud_cleanup(bool_t killreader)
{
	DBGPRN(DBG_DEVIO)(errfp,
			  "\nossaud_cleanup: Cleaning up writer pid=%d\n",
			  (int) getpid());

	if (oss_audio_fd >= 0)
		(void) ossaud_close_dev();

	if (oss_file_fd >= 0) {
		gen_close_file(oss_wcd, oss_file_fd, oss_file_fmt);
		oss_file_fd = -1;
	}

	if (oss_pipe_fd >= 0) {
		gen_close_pipe(oss_wcd, oss_pipe_fd);
		oss_pipe_fd = -1;
	}

	(void) sleep(1);

	if (oss_pipe_pid > 0) {
		(void) kill(oss_pipe_pid, SIGTERM);
		oss_pipe_pid = -1;
	}

	if (oss_wcd != NULL) {
		if (oss_wcd->i != NULL) {
			if (killreader && oss_wcd->i->reader > 0) {
				(void) kill(oss_wcd->i->reader, SIGTERM);
				oss_wcd->i->reader = 0;
				oss_wcd->i->state = CDSTAT_COMPLETED;
			}

			/* Reset frames played counter */
			oss_wcd->i->frm_played = 0;
		}

		MEM_FREE(oss_wcd);
		oss_wcd = NULL;
	}
}


/*
 * ossaud_onterm
 *	SIGTERM and SIGPIPE signal handler - gracefully exit
 *
 * Args:
 *	signo - The signal number
 *
 * Return:
 *	Nothing.
 */
STATIC void
ossaud_onterm(int signo)
{
	ossaud_cleanup(signo == SIGPIPE);
	exit(1);
}


/*
 * oss_writeinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
oss_writeinit(void)
{
	return (CDDA_WRITEDEV | CDDA_WRITEFILE | CDDA_WRITEPIPE);
}


/*
 * oss_write
 *	Opens audio device/file, attaches shared memory and semaphores.
 *	Continuously reads data from shared memory and writes it
 *	to the audio device/file.
 *
 * Args:
 *	mode - play mode (bitmask PLAYMODE_CDDA, etc.)
 *	fmt  - output file format
 *	s    - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
oss_write(byte_t mode, byte_t fmt, curstat_t *s)
{
	word32_t	estlen;
	int		ret,
			wstat;

	if (!PLAYMODE_IS_CDDA(mode)) {
		(void) fprintf(errfp, "oss_write: Nothing to do.\n");
		return FALSE;
	}

	if ((mode & PLAYMODE_CDDA) != 0) {
		oss_write_dev = TRUE;
	}
	if ((mode & PLAYMODE_FILE) != 0) {
		oss_write_file = TRUE;
	}
	if ((mode & PLAYMODE_PIPE) != 0) {
		oss_write_pipe = TRUE;
		oss_pipeprog = s->pipeprog;
	}
	oss_file_fmt = fmt;

	/* Generic services initialization */
	gen_write_init();

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);
	(void) signal(SIGALRM, SIG_IGN);
	(void) signal(SIGUSR1, SIG_IGN);
	(void) signal(SIGPIPE, ossaud_onterm);
	(void) signal(SIGTERM, ossaud_onterm);

	/* Allocate memory */
	oss_wcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (oss_wcd == NULL) {
		(void) fprintf(errfp, "oss_write: out of memory\n");
		return FALSE;
	}

	(void) memset(oss_wcd, 0, sizeof(cd_state_t));

	/* Initialize shared memory and semaphores */
	if ((oss_wsemid = cdda_rw_initipc(oss_wcd)) < 0) {
		ossaud_cleanup(TRUE);
		return FALSE;
	}

	estlen = (oss_wcd->i->end_lba - oss_wcd->i->start_lba + 1) *
		 CDDA_BLKSZ;

	if (oss_write_file || oss_write_pipe) {
		/* Open file and/or pipe */
		if (!app_data.cdda_trkfile && oss_write_file) {
			oss_file_fd = gen_open_file(
				oss_wcd,
				s->trkinfo[0].outfile,
				O_WRONLY | O_TRUNC | O_CREAT,
				S_IRUSR | S_IWUSR,
				oss_file_fmt, estlen
			);
			if (oss_file_fd < 0) {
				ossaud_cleanup(TRUE);
				return FALSE;
			}

			DBGPRN(DBG_DEVIO)(errfp,
					  "\nossaud_write: Writing file [%s]: "
					  "chunk_blks=%d\n",
					  s->trkinfo[0].outfile,
					  oss_wcd->cds->chunk_blocks);
		}

		if (oss_write_pipe) {
			oss_pipe_fd = gen_open_pipe(oss_wcd, oss_pipeprog,
						    &oss_pipe_pid,
						    oss_file_fmt, estlen);
			if (oss_pipe_fd < 0) {
				ossaud_cleanup(TRUE);
				return FALSE;
			}
			DBGPRN(DBG_DEVIO)(errfp,
					  "\nossaud_write: Pipe to [%s]: "
					  "chunk_blks=%d\n",
					  oss_pipeprog,
					  oss_wcd->cds->chunk_blocks);
		}

		switch ((int) oss_file_fmt) {
		case FILEFMT_AU:
		case FILEFMT_AIFF:
		case FILEFMT_AIFC:
			oss_file_be = TRUE;
			break;
		case FILEFMT_WAV:
		default:
			oss_file_be = FALSE;
			break;
		}
	}

	if (oss_write_dev) {
		/* Open audio device */
		if (!ossaud_open_dev()) {
			ossaud_cleanup(TRUE);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
				  "\nossaud_write: Writing device [%s]: "
				  "chunk_blks=%d\n",
				  oss_device, oss_wcd->cds->chunk_blocks);

		/* Configure audio */
		if (!ossaud_config()) {
			ossaud_cleanup(TRUE);
			return FALSE;
		}

		/* Initial settings */
		ossaud_update();
	}

	/* Do the writing */
	if (!ossaud_write(s)) {
		ossaud_cleanup(TRUE);
		return FALSE;
	}

	if (oss_write_file) {
		gen_close_file(oss_wcd, oss_file_fd, oss_file_fmt);
		oss_file_fd = -1;
	}

	ret = 0;
	if (oss_write_pipe) {
		gen_close_pipe(oss_wcd, oss_pipe_fd);
		oss_pipe_fd = -1;

		while ((ret = WAITPID(oss_pipe_pid, &wstat, 0)) !=
			oss_pipe_pid) {
			if (ret < 0 && errno != EINTR)
				break;
		}

		if (WIFEXITED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp,
					  "Pipe program exited, status %d\n",
					  WEXITSTATUS(wstat));
		}
		else if (WIFSIGNALED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp,
					  "Pipe program signal, status %d\n",
					  WTERMSIG(wstat));
		}

		oss_pipe_pid = -1;
	}

	ossaud_cleanup((bool_t) (ret != 0));
	return TRUE;
}

#endif	/* CDDA_WR_OSS CDDA_SYSVIPC */

