/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   SCSI pass-through CDDA read support
 */
#ifndef LINT
static char *_scsiread_c_ident_ = "@(#)scsiread.c	7.26 02/05/17";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_RD_SCSIPT) && defined(CDDA_SYSVIPC)

#include "libdi_d/scsipt.h"
#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/scsiread.h"

extern appdata_t	app_data;
extern FILE		*errfp;
extern void		*cdda_shmaddr;

STATIC int		spt_rsemid,	/* Semaphores identifier */
			spt_rfd;	/* CD device file descriptor */
STATIC cd_state_t	*spt_rcd;	/* CD state pointer */
STATIC byte_t		spt_cdb[MAX_CMDLEN];
STATIC bool_t		spt_modesel = FALSE;
					/* Did a mode select to enable CDDA */


/*
 * scsicd_read_fillbuf
 *	Before beginning to play we fill the data buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
scsicd_read_fillbuf(int fd)
{
	int			i,
				cmdlen;
	byte_t			*start,
				*end,
				*p;
	mode_sense_6_data_t	*ms_data6 = NULL;
	mode_sense_10_data_t	*ms_data10 = NULL;
	blk_desc_t		*bdesc;
	scsi_readaudio_t	cdda;
	byte_t			buf[SZ_MSENSE];
	bool_t			ret;

	/* Get lock */
	cdda_waitsem(spt_rsemid, LOCK);

	/* Initialize */
	start = &spt_rcd->cdb->data[spt_rcd->cds->olap_bytes];

	if (app_data.cdda_modesel) {
		(void) memset(buf, 0, sizeof(buf));

		if (app_data.msen_10) {
			ms_data10 = (mode_sense_10_data_t *)(void *) buf;
			ms_data10->bdescr_len =	
				util_bswap16(sizeof(blk_desc_t));
			bdesc = (blk_desc_t *)(void *) ms_data10->data;
		}
		else {
			ms_data6 = (mode_sense_6_data_t *)(void *) buf;
			ms_data6->bdescr_len = sizeof(blk_desc_t);
			bdesc = (blk_desc_t *)(void *) ms_data6->data;
		}

		/* Set Density code for CDDA */
		bdesc->dens_code = (byte_t) app_data.cdda_scsidensity;

		/* Block size for CDDA */
		bdesc->blk_len = util_bswap24(CDDA_BLKSZ);

		/* Write new setting */
		if (!scsipt_modesel(fd, buf, 0, 0)) {
			(void) sprintf(spt_rcd->i->msgbuf,
				"scsicd_read_fillbuf: Mode select failed.");
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", spt_rcd->i->msgbuf);
			cdda_postsem(spt_rsemid, LOCK);
			return FALSE;
		}

		spt_modesel = TRUE;
	}

	/* Set up for read */
	cdda.pos.logical = spt_rcd->i->start_lba;
	cdda.nframes = spt_rcd->cds->chunk_blocks + spt_rcd->cds->olap_blocks;
	cdda.buf = (u_char *) start;
	cdda.addrfmt = READFMT_LBA;

	cdda_rd_setcurtrk(spt_rcd, (int) cdda.pos.logical);

	/* Fill up our buffer */
	i = 0;

	while (i < spt_rcd->cds->buffer_chunks && !spt_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * spt_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (cdda.pos.logical + spt_rcd->cds->chunk_blocks >=
		    spt_rcd->i->end_lba) {
			cdda.nframes = spt_rcd->i->end_lba - cdda.pos.logical;
			(void) memset(
				spt_rcd->cdb->data,
				0,
				spt_rcd->cds->chunk_bytes +
				(spt_rcd->cds->olap_bytes << 1)
			);
			spt_rcd->i->cdda_done = 1;
		}

		SCSICDB_RESET(spt_cdb);

		spt_cdb[2] = (cdda.pos.logical >> 24) & 0xff;
		spt_cdb[3] = (cdda.pos.logical >> 16) & 0xff;
		spt_cdb[4] = (cdda.pos.logical >>  8) & 0xff;
		spt_cdb[5] = cdda.pos.logical & 0xff;

		switch (app_data.cdda_scsireadcmd) {
		case CDDA_MMC:
		    /* MMC Read CD (12) */
		    cmdlen = 12;
		    spt_cdb[0]  = OP_L_READCD;
		    spt_cdb[6]  = (cdda.nframes >> 16) & 0xff;
		    spt_cdb[7]  = (cdda.nframes >>  8) & 0xff;
		    spt_cdb[8]  = cdda.nframes & 0xff;
		    spt_cdb[9]  = 0xf8;	/* Specify SYNC, EDC and ECC */
		    break;

		case CDDA_STD:
		    /* SCSI Read (10) */
		    cmdlen = 10;
		    spt_cdb[0]  = OP_M_READ;
		    spt_cdb[7]  = (cdda.nframes >> 8) & 0xff;
		    spt_cdb[8]  = cdda.nframes & 0xff;
		    break;

		case CDDA_NEC:
		    /* NEC Read CD-DA (10) */
		    cmdlen = 10;
		    spt_cdb[0]  = OP_VN_READCDDA;
		    spt_cdb[7]  = (cdda.nframes >> 8) & 0xff;
		    spt_cdb[8]  = cdda.nframes & 0xff;
		    break;

		case CDDA_SONY:
		    /* Sony Read CD-DA (12) */
		    cmdlen = 12;
		    spt_cdb[0]  = OP_VS_READCDDA;
		    spt_cdb[6]  = (cdda.nframes >> 24) & 0xff;
		    spt_cdb[7]  = (cdda.nframes >> 16) & 0xff;
		    spt_cdb[8]  = (cdda.nframes >>  8) & 0xff;
		    spt_cdb[9]  = cdda.nframes & 0xff;
		    spt_cdb[10] = 0x00;	/* No subcode */
		    break;

		default:
		    /* Invalid configuration */
		    (void) sprintf(spt_rcd->i->msgbuf,
			    "scsicd_read_fillbuf: Invalid cddaScsiReadCommand "
			    "parameter (%d)",
			    app_data.cdda_scsireadcmd);
		    DBGPRN(DBG_DEVIO)(errfp, "%s\n", spt_rcd->i->msgbuf);
		    cdda_postsem(spt_rsemid, LOCK);
		    return FALSE;
		}

		/* Send read command */
		ret = pthru_send(
			fd,
			spt_cdb,
			cmdlen,
			cdda.buf,
			cdda.nframes * CDDA_BLKSZ,
			OP_DATAIN,
			5,
			(bool_t) ((app_data.debug & DBG_DEVIO) != 0)
		);
		if (!ret) {
			(void) sprintf(spt_rcd->i->msgbuf,
				"scsicd_read_fillbuf: SCSI command failed.");
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", spt_rcd->i->msgbuf);
			cdda_postsem(spt_rsemid, LOCK);
			return FALSE;
		}

		/* Do jitter correction */
		if (i == 0)
			p = &spt_rcd->cdb->data[spt_rcd->cds->olap_bytes];
		else
			p = cdda_rd_corrjitter(spt_rcd);

		/* Data end */
		end = p + spt_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (spt_rcd->i->jitter) {
			(void) memcpy(
				spt_rcd->cdb->olap,
				end - spt_rcd->cds->search_bytes,
				spt_rcd->cds->search_bytes << 1
			);
		}

		/* Copy in */
		(void) memcpy(
			&spt_rcd->cdb->b[
			    spt_rcd->cds->chunk_bytes * spt_rcd->cdb->nextin
			],
			p,
			spt_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		spt_rcd->cdb->nextin++;
		spt_rcd->cdb->nextin %= spt_rcd->cds->buffer_chunks;
		spt_rcd->cdb->occupied++;

		cdda.pos.logical += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.pos.logical++;

		cdda_rd_setcurtrk(spt_rcd, (int) cdda.pos.logical);

		i++;

		/* Update debug level */
		app_data.debug = spt_rcd->i->debug;
	}

	/* No room available now */
	cdda_waitsem(spt_rsemid, ROOM);

	/* Signal data available */
	cdda_postsem(spt_rsemid, DATA);

	/* Release lock */
	cdda_postsem(spt_rsemid, LOCK);

	return TRUE;
}


/*
 * scsicd_read_cont
 *	Function responsible for reading from the cd,
 *	correcting for jitter and writing to the buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
scsicd_read_cont(int fd)
{
	int			cmdlen;
	byte_t			*start,
				*end,
				*p;
	scsi_readaudio_t	cdda;
	bool_t			ret;

	/* Initialize */
	start = &spt_rcd->cdb->data[spt_rcd->cds->olap_bytes];

	/* Set up for read */
	cdda.pos.logical = spt_rcd->i->start_lba + spt_rcd->cds->buffer_blocks;
	cdda.nframes = spt_rcd->cds->chunk_blocks + spt_rcd->cds->olap_blocks;
	cdda.buf = (u_char *) start;
	cdda.addrfmt = READFMT_LBA;

	cdda_rd_setcurtrk(spt_rcd, (int) cdda.pos.logical);

	/* While not stopped or finished */
	while (spt_rcd->i->state != CDSTAT_COMPLETED &&
	       !spt_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * spt_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (cdda.pos.logical + spt_rcd->cds->chunk_blocks >=
		    spt_rcd->i->end_lba) {
			cdda.nframes = spt_rcd->i->end_lba - cdda.pos.logical;
			(void) memset(
				spt_rcd->cdb->data,
				0,
				spt_rcd->cds->chunk_bytes +
				(spt_rcd->cds->olap_bytes << 1)
			);
			spt_rcd->i->cdda_done = 1;
		}

		SCSICDB_RESET(spt_cdb);

		spt_cdb[2] = (cdda.pos.logical >> 24) & 0xff;
		spt_cdb[3] = (cdda.pos.logical >> 16) & 0xff;
		spt_cdb[4] = (cdda.pos.logical >>  8) & 0xff;
		spt_cdb[5] = cdda.pos.logical & 0xff;

		switch (app_data.cdda_scsireadcmd) {
		case CDDA_MMC:
		    /* MMC Read CD (12) */
		    cmdlen = 12;
		    spt_cdb[0]  = OP_L_READCD;
		    spt_cdb[6]  = (cdda.nframes >> 16) & 0xff;
		    spt_cdb[7]  = (cdda.nframes >>  8) & 0xff;
		    spt_cdb[8]  = cdda.nframes & 0xff;
		    spt_cdb[9]  = 0xf8;	/* Specify SYNC, EDC and ECC */
		    break;

		case CDDA_STD:
		    /* SCSI Read (10) */
		    cmdlen = 10;
		    spt_cdb[0]  = OP_M_READ;
		    spt_cdb[7]  = (cdda.nframes >> 8) & 0xff;
		    spt_cdb[8]  = cdda.nframes & 0xff;
		    break;

		case CDDA_NEC:
		    /* NEC Read CD-DA (10) */
		    cmdlen = 10;
		    spt_cdb[0]  = OP_VN_READCDDA;
		    spt_cdb[7]  = (cdda.nframes >> 8) & 0xff;
		    spt_cdb[8]  = cdda.nframes & 0xff;
		    break;

		case CDDA_SONY:
		    /* Sony Read CD-DA (12) */
		    cmdlen = 12;
		    spt_cdb[0]  = OP_VS_READCDDA;
		    spt_cdb[6]  = (cdda.nframes >> 24) & 0xff;
		    spt_cdb[7]  = (cdda.nframes >> 16) & 0xff;
		    spt_cdb[8]  = (cdda.nframes >>  8) & 0xff;
		    spt_cdb[9]  = cdda.nframes & 0xff;
		    spt_cdb[10] = 0x00;	/* No subcode */
		    break;

		default:
		    /* Invalid configuration */
		    (void) sprintf(spt_rcd->i->msgbuf,
			    "scsicd_read_cont: Invalid cddaScsiReadCommand "
			    "parameter (%d)",
			    app_data.cdda_scsireadcmd);
		    DBGPRN(DBG_DEVIO)(errfp, "%s\n", spt_rcd->i->msgbuf);
		    return FALSE;
		}

		/* Send read command */
		ret = pthru_send(
			fd,
			spt_cdb,
			cmdlen,
			cdda.buf,
			cdda.nframes * CDDA_BLKSZ,
			OP_DATAIN,
			5,
			(bool_t) ((app_data.debug & DBG_DEVIO) != 0)
		);
		if (!ret) {
			(void) sprintf(spt_rcd->i->msgbuf,
				"scsicd_read_cont: SCSI command failed.");
			return FALSE;
		}

		/* Do jitter correction */
		p = cdda_rd_corrjitter(spt_rcd);

		/* Data end */
		end = p + spt_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (spt_rcd->i->jitter) {
			(void) memcpy(
				spt_rcd->cdb->olap,
				end - spt_rcd->cds->search_bytes,
				spt_rcd->cds->search_bytes << 1
			);
		}

		/* Get lock */
		cdda_waitsem(spt_rsemid, LOCK);

		/* Wait until there is room */
		while (spt_rcd->cdb->occupied >= spt_rcd->cds->buffer_chunks &&
		       spt_rcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(spt_rsemid, LOCK);
			cdda_waitsem(spt_rsemid, ROOM);
			cdda_waitsem(spt_rsemid, LOCK);
		}

		/* Break if completed */
		if (spt_rcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(spt_rsemid, LOCK);
			break;
		}

		/* Copy to free location */
		(void) memcpy(
			&spt_rcd->cdb->b[
			    spt_rcd->cds->chunk_bytes * spt_rcd->cdb->nextin
			],
			p,
			spt_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		spt_rcd->cdb->nextin++;
		spt_rcd->cdb->nextin %= spt_rcd->cds->buffer_chunks;
		spt_rcd->cdb->occupied++;

		cdda.pos.logical += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.pos.logical++;

		cdda_rd_setcurtrk(spt_rcd, (int) cdda.pos.logical);

		/* Signal data available */
		cdda_postsem(spt_rsemid, DATA);

		/* Release lock */
		cdda_postsem(spt_rsemid, LOCK);

		/* Update debug level */
		app_data.debug = spt_rcd->i->debug;
	}

	/* Wake up writer */
	cdda_postsem(spt_rsemid, DATA);

	return TRUE;
}


/*
 * scsicd_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killwriter - Whether to kill writer process
 *
 * Return:
 *	Nothing.
 */
STATIC void
scsicd_cleanup(bool_t killwriter)
{
	mode_sense_6_data_t	*ms_data6 = NULL;
	mode_sense_10_data_t	*ms_data10 = NULL;
	blk_desc_t		*bdesc;
	byte_t			buf[SZ_MSENSE];

	DBGPRN(DBG_DEVIO)(errfp,
			  "\nscsicd_cleanup: Cleaning up reader pid=%d\n",
			  (int) getpid());

	/* Restore CD-ROM mode if necessary */
	if (spt_modesel && app_data.cdda_modesel) {
		(void) memset(buf, 0, sizeof(buf));

		if (app_data.msen_10) {
			ms_data10 = (mode_sense_10_data_t *)(void *) buf;
			ms_data10->bdescr_len =
				util_bswap16(sizeof(blk_desc_t));
			bdesc = (blk_desc_t *)(void *) ms_data10->data;
		}
		else {
			ms_data6 = (mode_sense_6_data_t *)(void *) buf;
			ms_data6->bdescr_len = sizeof(blk_desc_t);
			bdesc = (blk_desc_t *)(void *) ms_data6->data;
		}

		/* Restore density code */
		bdesc->dens_code = 0;

		/* Restore sector size */
		bdesc->blk_len = util_bswap24(app_data.drv_blksz);

		/* Write new setting */
		if (!scsipt_modesel(spt_rfd, buf, 0, 0)) {
			DBGPRN(DBG_DEVIO)(errfp,
				"scsicd_cleanup: Mode select failed.\n");
		}

		spt_modesel = FALSE;
	}

	(void) sleep(1);

	if (spt_rcd != NULL) {
		if (killwriter && spt_rcd->i != NULL &&
		    spt_rcd->i->writer > 0) {
			(void) kill(spt_rcd->i->writer, SIGTERM);
			spt_rcd->i->writer = 0;
			spt_rcd->i->state = CDSTAT_COMPLETED;
		}

		MEM_FREE(spt_rcd);
		spt_rcd = NULL;
	}
}


/*
 * scsicd_onterm
 *	SIGTERM signal handler - gracefully exit
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
scsicd_onterm(int signo)
{
	scsicd_cleanup(FALSE);
	exit(1);
}


/*
 * spt_readinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
spt_readinit(void)
{
	return CDDA_READAUDIO;
}


/*
 * spt_read
 *	Attaches to shared memory and semaphores. Continuously reads
 *	data from the cd and places into shared memory.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
spt_read(int fd)
{
	/* Check configuration */
	if (app_data.di_method != 0 /* DI_SCSIPT */) {
		(void) fprintf(errfp,
			"spt_read: Inconsistent deviceInterfaceMethod "
			"and cddaReadMethod parameters.  Aborting.\n");
		return FALSE;
	}

	/* Save file descriptor */
	spt_rfd = fd;

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);
	(void) signal(SIGTERM, scsicd_onterm);

	spt_rcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (spt_rcd == NULL) {
		(void) fprintf(errfp, "spt_read: out of memory\n");
		return FALSE;
	}

	(void) memset(spt_rcd, 0, sizeof(cd_state_t));

	/* Initialize cd pointers to point into shared memory */
	if ((spt_rsemid = cdda_rw_initipc(spt_rcd)) < 0) {
		scsicd_cleanup(TRUE);
		return FALSE;
	}

	DBGPRN(DBG_DEVIO)(errfp,
			  "\nspt_read: Reading CDDA: "
			  "chunk_blks=%d, olap_blks=%d\n",
			  spt_rcd->cds->chunk_blocks,
			  spt_rcd->cds->olap_blocks);

	/* Fill data buffer */
	if (!scsicd_read_fillbuf(fd)) {
		scsicd_cleanup(TRUE);
		return FALSE;
	}

	/* Keep reading */
	if (!scsicd_read_cont(fd)) {
		scsicd_cleanup(TRUE);
		return FALSE;
	}

	scsicd_cleanup(FALSE);
	return TRUE;
}

#endif	/* CDDA_RD_SCSIPT CDDA_SYSVIPC */

