/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   Sun Solaris support
 *
 *   This software fragment contains code that interfaces the
 *   application to the Solaris operating environment.  The name "Sun"
 *   and "Solaris" are used here for identification purposes only.
 *
 *   Contributing author: Darragh O'Brien <darragh.obrien@sun.com>
 */
#ifndef LINT
static char *_solread_c_ident_ = "@(#)solread.c	7.50 02/04/09";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_RD_SOL) && defined(CDDA_SYSVIPC)

#include <stropts.h>
#include <sys/cdio.h>
#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/solread.h"

extern appdata_t	app_data;
extern FILE		*errfp;
extern void		*cdda_shmaddr;

STATIC int		sol_rsemid;	/* Semaphores identifier */
STATIC cd_state_t	*sol_rcd;	/* CD state pointer */


/*
 * solcd_read_fillbuf
 *	Before beginning to play we fill the data buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solcd_read_fillbuf(int fd)
{
	int			i;
	byte_t			*start,
				*end,
				*p;
	struct cdrom_cdda	cdda;

	/* Get lock */
	cdda_waitsem(sol_rsemid, LOCK);

	/* Initialize */
	start = &sol_rcd->cdb->data[sol_rcd->cds->olap_bytes];

	/* Set up for read */
	cdda.cdda_addr = sol_rcd->i->start_lba;
	cdda.cdda_length = sol_rcd->cds->chunk_blocks +
			   sol_rcd->cds->olap_blocks;
	cdda.cdda_subcode = CDROM_DA_NO_SUBCODE;
	cdda.cdda_data = (caddr_t) start;

	cdda_rd_setcurtrk(sol_rcd, (int) cdda.cdda_addr);

	/* Fill up our buffer */
	i = 0;

	while (i < sol_rcd->cds->buffer_chunks && !sol_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * sol_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (cdda.cdda_addr + sol_rcd->cds->chunk_blocks >=
		    sol_rcd->i->end_lba) {
			cdda.cdda_length =
				sol_rcd->i->end_lba - cdda.cdda_addr;
			(void) memset(
				sol_rcd->cdb->data,
				0,
				sol_rcd->cds->chunk_bytes +
				(sol_rcd->cds->olap_bytes << 1)
			);
			sol_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDROMCDDA, &cdda) < 0) {
			(void) sprintf(sol_rcd->i->msgbuf,
					"solcd_read_fillbuf: CDROMCDDA ioctl "
					"failed: %s",
					strerror(errno));
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_rcd->i->msgbuf);
			cdda_postsem(sol_rsemid, LOCK);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
				"\nSent CDROMCDDA ioctl: blk=%d len=%d\n",
				cdda.cdda_addr, cdda.cdda_length);

		/* Do jitter correction */
		if (i == 0)
			p = &sol_rcd->cdb->data[sol_rcd->cds->olap_bytes];
		else
			p = cdda_rd_corrjitter(sol_rcd);

		/* Data end */
		end = p + sol_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (sol_rcd->i->jitter) {
			(void) memcpy(
				sol_rcd->cdb->olap,
				end - sol_rcd->cds->search_bytes,
				sol_rcd->cds->search_bytes << 1
			);
		}

		/* Copy in */
		(void) memcpy(
			&sol_rcd->cdb->b[
			    sol_rcd->cds->chunk_bytes * sol_rcd->cdb->nextin
			],
			p,
			sol_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		sol_rcd->cdb->nextin++;
		sol_rcd->cdb->nextin %= sol_rcd->cds->buffer_chunks;
		sol_rcd->cdb->occupied++;

		cdda.cdda_addr += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.cdda_addr++;

		cdda_rd_setcurtrk(sol_rcd, (int) cdda.cdda_addr);

		i++;

		/* Update debug level */
		app_data.debug = sol_rcd->i->debug;
	}

	/* No room available now */
	cdda_waitsem(sol_rsemid, ROOM);

	/* Signal data available */
	cdda_postsem(sol_rsemid, DATA);

	/* Release lock */
	cdda_postsem(sol_rsemid, LOCK);

	return TRUE;
}


/*
 * solcd_read_cont
 *	Function responsible for reading from the cd,
 *	correcting for jitter and writing to the buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solcd_read_cont(int fd)
{
	byte_t			*start,
				*end,
				*p;
	struct cdrom_cdda	cdda;

	/* Initialize */
	start = &sol_rcd->cdb->data[sol_rcd->cds->olap_bytes];

	/* Set up for read */
	cdda.cdda_addr = sol_rcd->i->start_lba + sol_rcd->cds->buffer_blocks;
	cdda.cdda_length = sol_rcd->cds->chunk_blocks +
			   sol_rcd->cds->olap_blocks;
	cdda.cdda_data = (caddr_t) start;
	cdda.cdda_subcode = CDROM_DA_NO_SUBCODE;

	cdda_rd_setcurtrk(sol_rcd, (int) cdda.cdda_addr);

	/* While not stopped or finished */
	while (sol_rcd->i->state != CDSTAT_COMPLETED &&
	       !sol_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * sol_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if (cdda.cdda_addr + sol_rcd->cds->chunk_blocks >=
		    sol_rcd->i->end_lba) {
			cdda.cdda_length =
				sol_rcd->i->end_lba - cdda.cdda_addr;
			(void) memset(
				sol_rcd->cdb->data,
				0,
				sol_rcd->cds->chunk_bytes +
				(sol_rcd->cds->olap_bytes << 1)
			);
			sol_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDROMCDDA, &cdda) < 0) {
			(void) sprintf(sol_rcd->i->msgbuf,
					"solcd_read_cont: CDROMCDDA ioctl "
					"failed: %s",
					strerror(errno));
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_rcd->i->msgbuf);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
				"\nSent CDROMCDDA ioctl: blk=%d len=%d\n",
				cdda.cdda_addr, cdda.cdda_length);

		/* Do jitter correction */
		p = cdda_rd_corrjitter(sol_rcd);

		/* Data end */
		end = p + sol_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (sol_rcd->i->jitter) {
			(void) memcpy(
				sol_rcd->cdb->olap,
				end - sol_rcd->cds->search_bytes,
				sol_rcd->cds->search_bytes << 1
			);
		}

		/* Get lock */
		cdda_waitsem(sol_rsemid, LOCK);

		/* Wait until there is room */
		while (sol_rcd->cdb->occupied >= sol_rcd->cds->buffer_chunks &&
		       sol_rcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(sol_rsemid, LOCK);
			cdda_waitsem(sol_rsemid, ROOM);
			cdda_waitsem(sol_rsemid, LOCK);
		}

		/* Break if completed */
		if (sol_rcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(sol_rsemid, LOCK);
			break;
		}

		/* Copy to free location */
		(void) memcpy(
			&sol_rcd->cdb->b[
			    sol_rcd->cds->chunk_bytes * sol_rcd->cdb->nextin
			],
			p,
			sol_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		sol_rcd->cdb->nextin++;
		sol_rcd->cdb->nextin %= sol_rcd->cds->buffer_chunks;
		sol_rcd->cdb->occupied++;

		cdda.cdda_addr += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.cdda_addr++;

		cdda_rd_setcurtrk(sol_rcd, (int) cdda.cdda_addr);

		/* Signal data available */
		cdda_postsem(sol_rsemid, DATA);

		/* Release lock */
		cdda_postsem(sol_rsemid, LOCK);

		/* Update debug level */
		app_data.debug = sol_rcd->i->debug;
	}

	/* Wake up writer */
	cdda_postsem(sol_rsemid, DATA);

	return TRUE;
}


/*
 * solcd_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killwriter - Whether to kill the writer process
 *
 * Return:
 *	Nothing.
 */
STATIC void
solcd_cleanup(bool_t killwriter)
{
	DBGPRN(DBG_DEVIO)(errfp,
			  "\nsolcd_cleanup: Cleaning up reader pid=%d\n",
			  (int) getpid());

	(void) sleep(1);

	if (sol_rcd != NULL) {
		if (killwriter && sol_rcd->i != NULL &&
		    sol_rcd->i->writer > 0) {
			(void) kill(sol_rcd->i->writer, SIGTERM);
			sol_rcd->i->writer = 0;
			sol_rcd->i->state = CDSTAT_COMPLETED;
		}

		MEM_FREE(sol_rcd);
		sol_rcd = NULL;
	}
}


/*
 * solcd_onterm
 *	SIGTERM signal handler - gracefully exit
 *
 * Args:
 *	signo - The signal number.
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
solcd_onterm(int signo)
{
	solcd_cleanup(FALSE);
	exit(1);
}


/*
 * sol_readinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
sol_readinit(void)
{
	return CDDA_READAUDIO;
}


/*
 * sol_read
 *	Attaches to shared memory and semaphores. Continuously reads
 *	data from the cd and places into shared memory.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
sol_read(int fd)
{
	struct sigaction	action_term;

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);

	action_term.sa_handler = solcd_onterm;
	sigemptyset(&action_term.sa_mask);
	action_term.sa_flags = 0;
	if (sigaction(SIGTERM, &action_term, NULL) < 0) {
		(void) fprintf(errfp,
				"sol_read: sigaction failed (SIGTERM): %s\n",
				strerror(errno));
		return FALSE;
	}

	/* Allocate mempry */
	sol_rcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (sol_rcd == NULL) {
		(void) fprintf(errfp, "sol_read: out of memory\n");
		return FALSE;
	}

	(void) memset(sol_rcd, 0, sizeof(cd_state_t));

	/* Initialize cd pointers to point into shared memory */
	if ((sol_rsemid = cdda_rw_initipc(sol_rcd)) < 0) {
		solcd_cleanup(TRUE);
		return FALSE;
	}

	DBGPRN(DBG_DEVIO)(errfp,
			  "\nsolcd_read: Reading CDDA: "
			  "chunk_blks=%d, olap_blks=%d\n",
			  sol_rcd->cds->chunk_blocks,
			  sol_rcd->cds->olap_blocks);

	/* Fill data buffer */
	if (!solcd_read_fillbuf(fd)) {
		solcd_cleanup(TRUE);
		return FALSE;
	}

	/* Keep reading */
	if (!solcd_read_cont(fd)) {
		solcd_cleanup(TRUE);
		return FALSE;
	}

	solcd_cleanup(FALSE);
	return TRUE;
}

#endif	/* CDDA_RD_SOL CDDA_SYSVIPC */

