/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   Sun Solaris support
 *
 *   This software fragment contains code that interfaces the
 *   application to the Solaris operating environment.  The name "Sun"
 *   and "Solaris" are used here for identification purposes only.
 *
 *   Contributing author: Darragh O'Brien <darragh.obrien@sun.com>
 */
#ifndef LINT
static char *_solwrite_c_ident_ = "@(#)solwrite.c	7.94 02/04/23";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_WR_SOL) && defined(CDDA_SYSVIPC)

#include <stropts.h>
#include <sys/stream.h>
#include <sys/audioio.h>
#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/genwrite.h"
#include "cdda_d/solwrite.h"

extern appdata_t	app_data;
extern FILE		*errfp;

STATIC int		sol_audio_fd = -1,	/* Audio device fd */
			sol_file_fd = -1,	/* Output file fd */
			sol_pipe_fd = -1,	/* Pipe to program fd */
			sol_wsemid;		/* Semaphores identifier */
STATIC pid_t		sol_pipe_pid = -1;	/* Pipe to program pid */
STATIC cd_state_t	*sol_wcd;		/* Shared memory pointer */
STATIC char		*sol_device,		/* Audio device */
			*sol_pipeprog;		/* Pipe to program */
STATIC bool_t		sol_write_dev = FALSE,	/* Write to audio device */
			sol_write_file = FALSE,	/* Write to output file */
			sol_write_pipe = FALSE,	/* Pipe to program */
			sol_file_be = FALSE;	/* Big endian output file */
STATIC byte_t		sol_file_fmt;		/* Output file format */


/*
 * solaud_open
 *	Wrapper function for open(2) with retries for EINTR.
 *
 * Args:
 *	Same as open(2)
 *
 * Return:
 *	< 0   - open failed
 *	>= 0  - open successful
 */
STATIC int
solaud_open(char *path, int oflag, mode_t mode)
{
	int	fd,
		cnt = 0;

	do {
		errno = 0;
		if ((fd = open(path, oflag, mode)) >= 0)
			return (fd);
	} while (errno == EINTR && ++cnt < CDDA_INTR_MAX);

	return -1;
}


/*
 * solaud_close
 *	Wrapper function for close(2) with retries for EINTR.
 *
 * Args:
 *	Same as close(2)
 *
 * Return:
 *	FALSE - close failed
 *	TRUE  - close successful
 */
STATIC bool_t
solaud_close(int fd)
{
	int	cnt = 0;

	do {
		errno = 0;
		if (close(fd) == 0)
			return TRUE;
	} while (errno == EINTR && ++cnt < CDDA_INTR_MAX);

	return FALSE;
}


/*
 * solaud_ioctl
 *	Wrapper function for ioctl(2) with retries for EINTR.
 *
 * Args:
 *	Same as ioctl(2)
 *
 * Return:
 *	FALSE - ioctl failed
 *	TRUE  - ioctl successful
 */
STATIC bool_t
solaud_ioctl(int fd, int req, void *arg)
{
	int	cnt = 0;

	do {
		errno = 0;
		if (ioctl(fd, req, arg) >= 0)
			return TRUE;
	} while (errno == EINTR && ++cnt < CDDA_INTR_MAX);

	return FALSE;
}


/*
 * solaud_scale_vol
 *	Scale volume from xmcd 0-100 to Solaris audio device 0-AUDIO_MAX_GAIN
 *
 * Args:
 *	vol - xmcd volume 0-100
 *
 * Return:
 *	Solaris volume 0-AUDIO_MAX_GAIN
 */
STATIC unsigned int
solaud_scale_vol(int vol)
{
	unsigned int	ret;

	ret = (vol * AUDIO_MAX_GAIN) / 100;
	if (((vol * AUDIO_MAX_GAIN) % 100) >= 50)
		ret++;

	/* Range checking */
	if (ret > AUDIO_MAX_GAIN)
		ret = AUDIO_MAX_GAIN;

	return (ret);
}


/*
 * solaud_unscale_vol
 *	Scale volume from Solaris audio device 0-AUDIO_MAX_GAIN to xmcd 0-100
 *
 * Args:
 *	vol - Solaris volume 0-AUDIO_MAX_GAIN
 *
 * Return:
 *	xmcd volume 0-100
 */
STATIC int
solaud_unscale_vol(unsigned int vol)
{
	int	ret;

	ret = (vol * 100) / AUDIO_MAX_GAIN;
	if (((vol * 100) % AUDIO_MAX_GAIN) >= 128)
		ret++;

	/* Range checking */
	if (ret < 0)
		ret = 0;
	else if (ret > 100)
		ret = 100;

	return (ret);
}


/*
 * solaud_scale_bal
 *	Convert xmcd individual channel volume percentage values to
 *	Solaris audio balance value.
 *
 * Args:
 *	vol_left  - left volume percentage
 *	vol_right - right volume percentage
 *
 * Return:
 *	Solaris balance 0-AUDIO_RIGHT_BALANCE
 */
STATIC uchar_t
solaud_scale_bal(int vol_left, int vol_right)
{
	uchar_t	ret;

	if (vol_left == 100 && vol_right == 100) {
		ret = AUDIO_MID_BALANCE;
	}
	else if (vol_right < 100) {
		ret = (vol_right * AUDIO_MID_BALANCE) / 100;
		if (((vol_right * AUDIO_MID_BALANCE) % 100) >= 50)
			ret++;
	}
	else if (vol_left < 100) {
		ret = ((vol_left * AUDIO_MID_BALANCE) / 100) +
			AUDIO_MID_BALANCE;
		if (((vol_left * AUDIO_MID_BALANCE) % 100) >= 50)
			ret++;
	}

	/* Range checking */
	if (ret > AUDIO_RIGHT_BALANCE)
		ret = AUDIO_RIGHT_BALANCE;

	return (ret);
}


/*
 * solaud_unscale_bal
 *	Convert Solaris audio balance value to xmcd individual channel
 *	volume percentage values.
 *
 * Args:
 *	bal       - Solaris audio balance value
 *	vol_left  - left volume percentage (return)
 *	vol_right - right volume percentage (return)
 *
 * Return:
 *	Nothing.
 */
STATIC void
solaud_unscale_bal(uchar_t bal, int *vol_left, int *vol_right)
{
	if (bal > AUDIO_RIGHT_BALANCE)
		bal = AUDIO_RIGHT_BALANCE;

	if (bal == AUDIO_MID_BALANCE) {
		*vol_left = 100;
		*vol_right = 100;
	}
	else if (bal < AUDIO_MID_BALANCE) {
		*vol_left = 100;
		*vol_right = (bal * 100) / AUDIO_MID_BALANCE;
		if (((bal * 100) % AUDIO_MID_BALANCE) >= 128)
			(*vol_right)++;
	}
	else if (bal > AUDIO_MID_BALANCE) {
		bal -= AUDIO_MID_BALANCE;
		*vol_left = (bal * 100) / AUDIO_MID_BALANCE;
		*vol_right = 100;
		if (((bal * 100) % AUDIO_MID_BALANCE) >= 128)
			(*vol_left)++;
	}

	/* Range checking */
	if (*vol_left < 0)
		*vol_left = 0;
	else if (*vol_left > 100)
		*vol_left = 100;

	if (*vol_right < 0)
		*vol_right = 0;
	else if (*vol_right > 100)
		*vol_right = 100;
}


/*
 * solaud_get_info
 *	Get audio hardware settings.
 *
 * Args:
 *	info - Info to return
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_get_info(audio_info_t *info)
{
	/* Retrieve settings */
	if (solaud_ioctl(sol_audio_fd, AUDIO_GETINFO, info) < 0) {
		(void) sprintf(sol_wcd->i->msgbuf,
			"solaud_get_info: AUDIO_GETINFO failed: %s",
			strerror(errno));
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		return FALSE;
	}

	return TRUE;
}


/*
 * solaud_set_info
 *	Set audio hardware settings.
 *
 * Args:
 *	info - Info to set
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_set_info(audio_info_t *info)
{
	/* Apply settings */
	if (solaud_ioctl(sol_audio_fd, AUDIO_SETINFO, info) < 0) {
		(void) sprintf(sol_wcd->i->msgbuf,
			"solaud_set_info: AUDIO_SETINFO failed: %s",
			strerror(errno));
		return FALSE;
	}

	return TRUE;
}


/*
 * solaud_flush
 *	Wrapper for STREAMS I_FLUSH ioctl.
 *
 * Args:
 *	fd - The audio device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_flush(int fd)
{
	/* Apply settings */
	if (solaud_ioctl(fd, I_FLUSH, (void *) FLUSHRW) < 0) {
		(void) sprintf(sol_wcd->i->msgbuf,
			"solaud_flush: I_FLUSH failed: %s",
			strerror(errno));
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		return FALSE;
	}

	return TRUE;
}


/*
 * solaud_drain
 *	Wrapper for AUDIO_DRAIN ioctl.
 *
 * Args:
 *	fd - Audio device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_drain(int fd)
{
	/* Apply settings */
	if (solaud_ioctl(fd, AUDIO_DRAIN, NULL) < 0) {
		(void) sprintf(sol_wcd->i->msgbuf,
				"solaud_drain: AUDIO_DRAIN failed: %s",
				strerror(errno));
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		return FALSE;
	}

	return TRUE;
}


/*
 * solaud_open_dev
 *	Opens the audio device.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_open_dev(void)
{
	char		*cp;
	struct stat	stbuf;

	if (!gen_set_eid(sol_wcd))
		return FALSE;

	if (sol_device != NULL) {
		MEM_FREE(sol_device);
		sol_device = NULL;
	}

	/* Use environment variable if set */
	if ((cp = getenv("AUDIODEV")) == NULL)
		cp = DEFAULT_DEV_AUDIO;

	if (!util_newstr(&sol_device, cp)) {
		(void) sprintf(sol_wcd->i->msgbuf,
				"solaud_open_dev: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		(void) gen_reset_eid(sol_wcd);
		return FALSE;
	}

	/* Check to make sure it's a device */
	if (stat(sol_device, &stbuf) < 0) {
		(void) sprintf(sol_wcd->i->msgbuf,
				"solaud_open_dev: stat of %s failed: %s",
				sol_device, strerror(errno));
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		(void) gen_reset_eid(sol_wcd);
		return FALSE;
	}
	if (!S_ISCHR(stbuf.st_mode)) {
		(void) sprintf(sol_wcd->i->msgbuf,
				"solaud_open_dev: %s is not a "
				"character device",
				sol_device);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		(void) gen_reset_eid(sol_wcd);
		return FALSE;
	}

	/* Open audio device */
	if ((sol_audio_fd = solaud_open(sol_device, O_WRONLY, 0)) < 0) {
		(void) sprintf(sol_wcd->i->msgbuf,
				"solaud_open_dev: open of %s failed: %s",
				sol_device, strerror(errno));
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		(void) gen_reset_eid(sol_wcd);
		return FALSE;
	}

	(void) gen_reset_eid(sol_wcd);
	return TRUE;
}


/*
 * solaud_close_dev
 *	Closes the audio device.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_close_dev(void)
{
	if (!gen_set_eid(sol_wcd))
		return FALSE;

	/* Close audio device */
	if (solaud_close(sol_audio_fd) < 0) {
		DBGPRN(DBG_DEVIO)(errfp,
			"solaud_close_dev: close failed: %s\n",
			strerror(errno));
		(void) gen_reset_eid(sol_wcd);
		return FALSE;
	}

	sol_audio_fd = -1;

	(void) gen_reset_eid(sol_wcd);
	return TRUE;
}


/*
 * solaud_config
 *	Sets up audio device for playing CD quality audio.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_config(void)
{
	audio_info_t	solaud_info;

	/* Initialize */
	SOL_AUDIO_INIT(&solaud_info, sizeof(audio_info_t));

	/* Play side */
	solaud_info.play.sample_rate = 44100;
	solaud_info.play.channels = 2;
	solaud_info.play.precision = 16;
	solaud_info.play.encoding = AUDIO_ENCODING_LINEAR;

	/* Apply new settings */
	if (!solaud_set_info(&solaud_info))
		return FALSE;

	/* Retrieve current settings */
	if (!solaud_get_info(&solaud_info))
		return FALSE;

	/* Update volume */
	sol_wcd->i->vol = util_untaper_vol(
		solaud_unscale_vol(solaud_info.play.gain)
	);

	/* Update balance */
	solaud_unscale_bal(solaud_info.play.balance,
			   &sol_wcd->i->vol_left, &sol_wcd->i->vol_right);

	return TRUE;
}


/*
 * solaud_write_eof
 *	Puts an eof marker in the audio stream.
 *
 * Args:
 *	fd - The audio device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_write_eof(int fd)
{
	int	cnt = 0;

	if (!gen_set_eid(sol_wcd))
		return FALSE;

	do {
		errno = 0;
		if (write(fd, 0, 0) >= 0) {
			(void) gen_reset_eid(sol_wcd);
			return TRUE;
		}

		/* Increment counter if not paused */
		if (sol_wcd->i->state != CDSTAT_PAUSED)
			cnt++;

	} while (errno == EINTR && cnt < CDDA_INTR_MAX);

	(void) sprintf(sol_wcd->i->msgbuf,
			"solaud_write_eof: write failed: %s",
			strerror(errno));
	DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);

	(void) gen_reset_eid(sol_wcd);
	return FALSE;
}


/*
 * solaud_write
 *	Function respsonsible for writing from the buffer to the audio
 *	device (and file if saving). Each chunk is followed by an eof
 *	marker.
 *
 * Args:
 *	s - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
solaud_write(curstat_t *s)
{
	byte_t	*in_data,
		*sw_data,
		*le_data,
		*be_data,
		*dev_data,
		*file_data,
		*pipe_data;
	char	*fname = NULL;
	int	trk_idx = -1;
	bool_t	ret;

	/* Get memory for audio device write buffer */
	le_data = (byte_t *) MEM_ALLOC("le_data", sol_wcd->cds->chunk_bytes);
	if (le_data == NULL) {
		(void) sprintf(sol_wcd->i->msgbuf,
				"solaud_write: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		return FALSE;
	}

	/* Get memory for audio file write buffer */
	be_data = (byte_t *) MEM_ALLOC("be_data", sol_wcd->cds->chunk_bytes);
	if (be_data == NULL) {
		(void) sprintf(sol_wcd->i->msgbuf,
				"solaud_write: out of memory");
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", sol_wcd->i->msgbuf);
		return FALSE;
	}

	(void) memset(le_data, 0, sol_wcd->cds->chunk_bytes);
	(void) memset(be_data, 0, sol_wcd->cds->chunk_bytes);

	in_data = app_data.cdda_bigendian ? be_data : le_data;
	sw_data = app_data.cdda_bigendian ? le_data : be_data;

	/* While not stopped */
	ret = TRUE;
	while (sol_wcd->i->state != CDSTAT_COMPLETED) {
		/* Get lock */
		cdda_waitsem(sol_wsemid, LOCK);

		/* End if finished and nothing in the buffer */
		if (sol_wcd->cdb->occupied == 0 && sol_wcd->i->cdda_done) {
			if (sol_write_dev && !solaud_drain(sol_audio_fd)) {
				ret = FALSE;
				break;
			}
			sol_wcd->i->state = CDSTAT_COMPLETED;
			cdda_postsem(sol_wsemid, LOCK);
			break;
		}

		/* Wait until there is something there */
		while (sol_wcd->cdb->occupied <= 0 &&
		       sol_wcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(sol_wsemid, LOCK);
			cdda_waitsem(sol_wsemid, DATA);
			cdda_waitsem(sol_wsemid, LOCK);
		}

		/* Break if stopped */
		if (sol_wcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(sol_wsemid, LOCK);
			break;
		}

		/* Copy a chunk from the nextout slot into the data buffers */
		(void) memcpy(
			in_data,
			&sol_wcd->cdb->b[
			    sol_wcd->cds->chunk_bytes * sol_wcd->cdb->nextout
			],
			sol_wcd->cds->chunk_bytes
		);

		/* Update pointers */
		sol_wcd->cdb->nextout++;
		sol_wcd->cdb->nextout %= sol_wcd->cds->buffer_chunks;
		sol_wcd->cdb->occupied--;

		/* Signal room available */
		cdda_postsem(sol_wsemid, ROOM);

		/* Release lock */
		cdda_postsem(sol_wsemid, LOCK);

		/* Change channel routing if necessary */
		gen_chroute(sol_wcd->i->chroute, (sword16_t *)(void *) in_data,
			    (size_t) sol_wcd->cds->chunk_bytes);

		/* Prepare byte-swapped version of the data */
		gen_byteswap(in_data, sw_data,
			     (size_t) sol_wcd->cds->chunk_bytes);

		/* Write to device */
		if (sol_write_dev) {
#if _BYTE_ORDER_ == _B_ENDIAN_
			dev_data = be_data;
#else
			dev_data = le_data;
#endif

			if (!gen_write_chunk(sol_wcd, sol_audio_fd,
					sol_device, dev_data,
					(size_t) sol_wcd->cds->chunk_bytes)) {
				sol_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(sol_wsemid, LOCK);
				ret = FALSE;
				break;
			}

			if (!solaud_write_eof(sol_audio_fd)) {
				sol_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(sol_wsemid, LOCK);
				ret = FALSE;
				break;
			}
		}

		/* Write to file */
		if (sol_write_file) {
			file_data = sol_file_be ? be_data : le_data;

			/* Open new track file if necessary */
			if (!app_data.cdda_trkfile) {
				fname = s->trkinfo[0].outfile;
			}
			else if (trk_idx != sol_wcd->i->trk_idx) {
				trk_idx = sol_wcd->i->trk_idx;

				if (sol_file_fd >= 0)
					gen_close_file(
						sol_wcd,
						sol_file_fd,
						sol_file_fmt
					);

				fname = s->trkinfo[trk_idx].outfile;
				sol_file_fd = gen_open_file(
					sol_wcd,
					fname,
					O_WRONLY | O_TRUNC | O_CREAT,
					S_IRUSR | S_IWUSR,
					sol_file_fmt,
					sol_wcd->i->trk_len
				);
				if (sol_file_fd < 0) {
					sol_wcd->i->state = CDSTAT_COMPLETED;
					cdda_postsem(sol_wsemid, LOCK);
					ret = FALSE;
					break;
				}

				DBGPRN(DBG_DEVIO)(errfp,
					  "\nflaud_write: Writing file [%s]: "
					  "chunk_blks=%d\n",
					  fname, sol_wcd->cds->chunk_blocks);
			}

			if (!gen_write_chunk(sol_wcd, sol_file_fd,
					fname, file_data,
					(size_t) sol_wcd->cds->chunk_bytes)) {
				sol_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(sol_wsemid, LOCK);
				ret = FALSE;
				break;
			}
		}

		/* Pipe to program */
		if (sol_write_pipe) {
			pipe_data = sol_file_be ? be_data : le_data;

			if (!gen_write_chunk(sol_wcd, sol_pipe_fd,
					sol_pipeprog, pipe_data,
					(size_t) sol_wcd->cds->chunk_bytes)) {
				sol_wcd->i->state = CDSTAT_COMPLETED;
				cdda_postsem(sol_wsemid, LOCK);
				ret = FALSE;
				break;
			}
		}

		if (!sol_write_dev)
			sol_wcd->i->frm_played += sol_wcd->cds->chunk_blocks;

		/* Update debug level */
		app_data.debug = sol_wcd->i->debug;
	}

	/* Wake up reader */
	cdda_postsem(sol_wsemid, ROOM);

	/* Free memory */
	MEM_FREE(le_data);
	MEM_FREE(be_data);

	return (ret);
}


/*
 * solaud_update
 *	SIGUSR1 signal handler. Updates volume, balance and pause settings.
 *
 * Args:
 *	signo - The signal number
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
solaud_update(int signo)
{
	static audio_info_t	solaud_info;

	app_data.vol_taper = sol_wcd->i->vol_taper;

	/* Initialize */
	SOL_AUDIO_INIT(&solaud_info, sizeof(audio_info_t));

	/* Update volume */
	solaud_info.play.gain =
		solaud_scale_vol(util_taper_vol(sol_wcd->i->vol));

	/* Update balance */
	solaud_info.play.balance =
		solaud_scale_bal(sol_wcd->i->vol_left, sol_wcd->i->vol_right);

	/* Update paused */
	solaud_info.play.pause = (sol_wcd->i->state == CDSTAT_PAUSED) ? 1 : 0;

	/* Apply new settings */
	(void) solaud_set_info(&solaud_info);
}


/*
 * solaud_status
 *	SIGALRM signal handler. Updates volume, balance and
 *	eof count from device.
 *
 * Args:
 *	signo - The signal number
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
solaud_status(int signo)
{
	static audio_info_t	solaud_info;

	/* Retrieve current settings */
	if (!solaud_get_info(&solaud_info))
		return;

	/* Update volume */
	sol_wcd->i->vol = util_untaper_vol(
		solaud_unscale_vol(solaud_info.play.gain)
	);

	/* Update balance */
	solaud_unscale_bal(solaud_info.play.balance,
			   &sol_wcd->i->vol_left, &sol_wcd->i->vol_right);

	/* Update frames written count */
	sol_wcd->i->frm_played = solaud_info.play.eof * FRAME_PER_SEC;

	/* Register alarm */
	(void) alarm(1);
}


/*
 * solaud_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killreader - Whether to kill the reader process
 *
 * Return:
 *	Nothing.
 */
STATIC void
solaud_cleanup(bool_t killreader)
{
	DBGPRN(DBG_DEVIO)(errfp,
			  "\nsolaud_cleanup: Cleaning up writer pid=%d\n",
			  (int) getpid());

	/* Clear any alarms */
	alarm(0);

	if (sol_audio_fd >= 0)
		(void) solaud_close_dev();

	if (sol_file_fd >= 0) {
		gen_close_file(sol_wcd, sol_file_fd, sol_file_fmt);
		sol_file_fd = -1;
	}

	if (sol_pipe_fd >= 0) {
		gen_close_pipe(sol_wcd, sol_pipe_fd);
		sol_pipe_fd = -1;
	}

	(void) sleep(1);

	if (sol_pipe_pid > 0) {
		(void) kill(sol_pipe_pid, SIGTERM);
		sol_pipe_pid = -1;
	}

	if (sol_wcd != NULL) {
		if (sol_wcd->i != NULL) {
			if (killreader && sol_wcd->i->reader > 0) {
				(void) kill(sol_wcd->i->reader, SIGTERM);
				sol_wcd->i->reader = 0;
				sol_wcd->i->state = CDSTAT_COMPLETED;
			}

			/* Reset frames played counter */
			sol_wcd->i->frm_played = 0;
		}

		MEM_FREE(sol_wcd);
		sol_wcd = NULL;
	}
}


/*
 * solaud_onterm
 *	SIGTERM and SIGPIPE signal handler - gracefully exit
 *
 * Args:
 *	signo - The signal number
 *
 * Return:
 *	Nothing.
 */
STATIC void
solaud_onterm(int signo)
{
	solaud_cleanup(signo == SIGPIPE);
	exit(1);
}


/*
 * sol_writeinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
sol_writeinit(void)
{
	struct utsname	*up;

	up = util_get_uname();
	if (strcmp(up->machine, "sun4c") == 0) {
		DBGPRN(DBG_DEVIO)(errfp, "sol_writeinit: %s\n%s\n",
			"CD-quality stereo playback on sun4c architecture",
			"is not supported.");
		return (CDDA_WRITEFILE | CDDA_WRITEPIPE);
	}

	return (CDDA_WRITEDEV | CDDA_WRITEFILE | CDDA_WRITEPIPE);
}


/*
 * sol_write
 *	Opens audio device/file, attaches shared memory and semaphores.
 *	Continuously reads data from shared memory and writes it
 *	to the audio device/file.
 *
 * Args:
 *	mode - play mode (bitmask PLAYMODE_CDDA, etc.)
 *	fmt  - output file format
 *	s    - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
sol_write(byte_t mode, byte_t fmt, curstat_t *s)
{
	int			ret,
				wstat;
	word32_t		estlen;
	struct sigaction	action_usr1,
				action_alrm,
				action_pipe,
				action_term;

	if (!PLAYMODE_IS_CDDA(mode)) {
		(void) fprintf(errfp, "sol_write: Nothing to do.\n");
		return FALSE;
	}

	if ((mode & PLAYMODE_CDDA) != 0) {
		sol_write_dev = TRUE;
	}
	if ((mode & PLAYMODE_FILE) != 0) {
		sol_write_file = TRUE;
	}
	if ((mode & PLAYMODE_PIPE) != 0) {
		sol_write_pipe = TRUE;
		sol_pipeprog = s->pipeprog;
	}
	sol_file_fmt = fmt;

	/* Generic services initialization */
	gen_write_init();

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);
	(void) signal(SIGALRM, SIG_IGN);
	(void) signal(SIGUSR1, SIG_IGN);

	action_pipe.sa_handler = solaud_onterm;
	sigemptyset(&action_pipe.sa_mask);
	action_pipe.sa_flags = 0;
	if (sigaction(SIGPIPE, &action_pipe, NULL) < 0) {
		(void) fprintf(errfp,
				"sol_write: sigaction failed (SIGPIPE): %s\n",
				strerror(errno));
		return FALSE;
	}

	action_term.sa_handler = solaud_onterm;
	sigemptyset(&action_term.sa_mask);
	action_term.sa_flags = 0;
	if (sigaction(SIGTERM, &action_term, NULL) < 0) {
		(void) fprintf(errfp,
				"sol_write: sigaction failed (SIGTERM): %s\n",
				strerror(errno));
		return FALSE;
	}

	/* Allocate memory */
	sol_wcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (sol_wcd == NULL) {
		(void) fprintf(errfp, "sol_write: out of memory\n");
		return FALSE;
	}

	(void) memset(sol_wcd, 0, sizeof(cd_state_t));

	/* Initialize shared memory and semaphores */
	if ((sol_wsemid = cdda_rw_initipc(sol_wcd)) < 0) {
		solaud_cleanup(TRUE);
		return FALSE;
	}

	estlen = (sol_wcd->i->end_lba - sol_wcd->i->start_lba + 1) *
		 CDDA_BLKSZ;

	if (sol_write_file || sol_write_pipe) {
		/* Open file and/or pipe */
		if (!app_data.cdda_trkfile && sol_write_file) {
			sol_file_fd = gen_open_file(
				sol_wcd,
				s->trkinfo[0].outfile,
				O_WRONLY | O_TRUNC | O_CREAT,
				S_IRUSR | S_IWUSR,
				sol_file_fmt, estlen
			);
			if (sol_file_fd < 0) {
				solaud_cleanup(TRUE);
				return FALSE;
			}

			DBGPRN(DBG_DEVIO)(errfp,
					  "\nsolaud_write: Writing file [%s]: "
					  "chunk_blks=%d\n",
					  s->trkinfo[0].outfile,
					  sol_wcd->cds->chunk_blocks);
		}

		if (sol_write_pipe) {
			sol_pipe_fd = gen_open_pipe(sol_wcd, sol_pipeprog,
						    &sol_pipe_pid,
						    sol_file_fmt, estlen);
			if (sol_pipe_fd < 0) {
				solaud_cleanup(TRUE);
				return FALSE;
			}
			DBGPRN(DBG_DEVIO)(errfp,
					  "\nsolaud_write: Pipe to [%s]: "
					  "chunk_blks=%d\n",
					  sol_pipeprog,
					  sol_wcd->cds->chunk_blocks);
		}

		switch ((int) sol_file_fmt) {
		case FILEFMT_AU:
		case FILEFMT_AIFF:
		case FILEFMT_AIFC:
			sol_file_be = TRUE;
			break;
		case FILEFMT_WAV:
		default:
			sol_file_be = FALSE;
			break;
		}
	}

	if (sol_write_dev) {
		/* Open audio device */
		if (!solaud_open_dev()) {
			solaud_cleanup(TRUE);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
				  "\nsolaud_write: Writing device [%s]: "
				  "chunk_blks=%d\n",
				  sol_device, sol_wcd->cds->chunk_blocks);

		/* Configure audio */
		if (!solaud_config()) {
			solaud_cleanup(TRUE);
			return FALSE;
		}

		/* Initial settings */
		solaud_update(SIGUSR1);

		/* Install SIGUSR1 signal handler */
		action_usr1.sa_handler = solaud_update;
		sigemptyset(&action_usr1.sa_mask);
		action_usr1.sa_flags = 0;
		if (sigaction(SIGUSR1, &action_usr1, NULL) < 0) {
			(void) fprintf(errfp,
				"sol_write: sigaction failed (SIGUSR1): %s\n",
				strerror(errno));
			solaud_cleanup(TRUE);
			return FALSE;
		}

		/* Install SIGALRM signal handler */
		action_alrm.sa_handler = solaud_status;
		sigemptyset(&action_alrm.sa_mask);
		action_alrm.sa_flags = 0;
		if (sigaction(SIGALRM, &action_alrm, NULL) < 0) {
			(void) fprintf(errfp,
				"sol_write: sigaction failed (SIGALRM): %s\n",
				strerror(errno));
			solaud_cleanup(TRUE);
			return FALSE;
		}

		/* Register alarm */
		(void) alarm(1);
	}

	/* Do the writing */
	if (!solaud_write(s)) {
		solaud_cleanup(TRUE);
		return FALSE;
	}

	/* Flush audio */
	if (sol_write_dev) {
		if (!solaud_flush(sol_audio_fd)) {
			solaud_cleanup(TRUE);
			return FALSE;
		}
	}

	if (sol_write_file) {
		gen_close_file(sol_wcd, sol_file_fd, sol_file_fmt);
		sol_file_fd = -1;
	}

	if (sol_write_pipe) {
		gen_close_pipe(sol_wcd, sol_pipe_fd);
		sol_pipe_fd = -1;

		while ((ret = WAITPID(sol_pipe_pid, &wstat, 0)) !=
		       sol_pipe_pid) {
			if (ret < 0 && errno != EINTR)
				break;
		}

		if (WIFEXITED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp,
					  "Pipe program exited, status %d\n",
					  WEXITSTATUS(wstat));
		}
		else if (WIFSIGNALED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp,
					  "Pipe program signal, status %d\n",
					  WTERMSIG(wstat));
		}

		sol_pipe_pid = -1;
	}

	solaud_cleanup((bool_t) (ret != 0));
	return TRUE;
}

#endif	/* CDDA_WR_SOL CDDA_SYSVIPC */

