/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef LINT
static char *_sysvipc_c_ident_ = "@(#)sysvipc.c	7.79 02/04/10";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"
#include "cdda_d/sysvipc.h"


#ifdef CDDA_SYSVIPC

#include "cdda_d/scsiread.h"
#include "cdda_d/solread.h"
#include "cdda_d/lxread.h"
#include "cdda_d/fbread.h"
#include "cdda_d/aixread.h"
#include "cdda_d/osswrite.h"
#include "cdda_d/solwrite.h"
#include "cdda_d/flwrite.h"
#include <sys/sem.h>


extern appdata_t	app_data;
extern FILE		*errfp;
extern cdda_client_t	*cdda_clinfo;

void			*cdda_shmaddr;		/* Shared memory pointer */

STATIC int		shmid = -1;		/* Shared memory identifier */
STATIC int		semid = -1;		/* Semaphores identifier */
STATIC cd_state_t	*cd;			/* CD state pointer */
STATIC char		errbuf[ERR_BUF_SZ];	/* Error message buffer */


/* Call table to branch into a read-method.  The array index must
 * correspond to the cddaReadMethod parameter.
 */
struct {
	word32_t (*readinit)(void);
	bool_t	 (*read)(int);
} sysvipc_rd_calltbl[] = {
	{ NULL, NULL },			/* CDDA_RD_NONE */
	{ spt_readinit, spt_read },	/* CDDA_RD_SCSIPT */
	{ sol_readinit, sol_read },	/* CDDA_RD_SOL */
	{ lx_readinit, lx_read },	/* CDDA_RD_LX */
	{ fb_readinit, fb_read },	/* CDDA_RD_FB */
	{ aix_readinit, aix_read }	/* CDDA_RD_AIX */
};


/* Call table to branch into a write-method.  The array index must
 * correspond to the cddaWriteMethod parameter.
 */
struct {
	word32_t (*writeinit)(void);
	bool_t	 (*write)(byte_t, byte_t, curstat_t *);
} sysvipc_wr_calltbl[] = {
	{ NULL, NULL },			/* CDDA_WR_NONE */
	{ oss_writeinit, oss_write },	/* CDDA_WR_OSS */
	{ sol_writeinit, sol_write },	/* CDDA_WR_SOL */
	{ NULL, NULL },			/* CDDA_WR_IRIX: to be implemented */
	{ NULL, NULL },			/* CDDA_WR_HPUX: to be implemented */
	{ fl_writeinit, fl_write }	/* CDDA_WR_FILE */
};


/*
 * cdda_getsem
 *	Initialize semaphores, making lock available.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
cdda_getsem(void)
{
	semun_t	arg;

	/* Initialize semaphores */
	if ((semid = semget(SEMKEY, 3, IPC_PERMS|IPC_CREAT|IPC_EXCL)) < 0) {
		/* If the semaphores exist already, use them. This
		 * should not happen if the application previously
		 * exited correctly.
		 */
		if (errno == EEXIST) {
			if ((semid = semget(SEMKEY, 3,
					    IPC_PERMS|IPC_EXCL)) < 0) {
				(void) sprintf(errbuf,
					    "cdda_getsem: semget failed "
					    "(errno=%d)",
					    errno);
				CDDA_INFO(errbuf);
				DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
				return FALSE;
			}
		}
		else {
			(void) sprintf(errbuf,
				    "cdda_getsem: semget failed (errno=%d)",
				    errno);
			CDDA_INFO(errbuf);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
			return FALSE;
		}
	}

	/* Make lock available */
	arg.val = 1;
	if (semctl(semid, LOCK, SETVAL, arg) < 0) {
		(void) sprintf(errbuf,
			    "cdda_getsem: semctl SETVAL failed (errno=%d)",
			    errno);
		CDDA_INFO(errbuf);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		return FALSE;
	}

	return TRUE;
}


/*
 * cdda_getshm
 *	Initialize shared memory.  We first work out how much shared
 *	memory we require and fill in a temporary cd_size_t structure
 *	along the way. We then create the shared memory segment and
 *	set the fields of *cd pointing into it.
 *
 * Args:
 *	None.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
cdda_getshm(void)
{
	cd_size_t	*cd_tmp_size;

	/* Store sizes here temporarily */
	cd_tmp_size = (cd_size_t *) MEM_ALLOC(
		"cd_tmp_size", sizeof(cd_size_t)
	);
	if (cd_tmp_size == NULL) {
		CDDA_FATAL(app_data.str_nomemory);
		return FALSE;
	}

	(void) memset(cd_tmp_size, 0, sizeof(cd_size_t));

	/* Sizes */
	cd_tmp_size->str_length = DEF_CDDA_STR_LEN;
	cd_tmp_size->chunk_blocks = app_data.cdda_readchkblks;

	/* Sizes depend on jitter */
	if (app_data.cdda_jitter_corr) {
		cd_tmp_size->olap_blocks = DEF_CDDA_OLAP_BLKS;
		cd_tmp_size->search_blocks = DEF_CDDA_SRCH_BLKS;
	}
	else {
		cd_tmp_size->olap_blocks = 0;
		cd_tmp_size->search_blocks = 0;
	}

	/* Chunk bytes */
	cd_tmp_size->chunk_bytes = cd_tmp_size->chunk_blocks * CDDA_BLKSZ;

	/* Buffer chunks */
	cd_tmp_size->buffer_chunks =
		(FRAME_PER_SEC * DEF_CDDA_BUFFER_SECS) /
		 cd_tmp_size->chunk_blocks;
	/* Check if we need to round up */
	if (((FRAME_PER_SEC * DEF_CDDA_BUFFER_SECS) %
	     cd_tmp_size->chunk_blocks) != 0)
		cd_tmp_size->buffer_chunks++;

	/* Buffer blocks */
	cd_tmp_size->buffer_blocks =
		cd_tmp_size->buffer_chunks * cd_tmp_size->chunk_blocks;

	/* Buffer bytes */
	cd_tmp_size->buffer_bytes =
		cd_tmp_size->buffer_chunks * cd_tmp_size->chunk_bytes;

	/* Overlap bytes */
	cd_tmp_size->olap_bytes = cd_tmp_size->olap_blocks * CDDA_BLKSZ;

	/* Search */
	cd_tmp_size->search_bytes = cd_tmp_size->search_blocks * CDDA_BLKSZ;

	/* How much shared memory do we need? */
	cd_tmp_size->size =
		sizeof(cd_size_t) + sizeof(cd_info_t) + sizeof(cd_buffer_t);

	/* Add memory for overlap */
	cd_tmp_size->size += (cd_tmp_size->search_bytes << 1);

	/* Add memory for data read */
	cd_tmp_size->size +=
		cd_tmp_size->chunk_bytes + (cd_tmp_size->olap_bytes << 1);

	/* Add memory for buffer */
	cd_tmp_size->size += cd_tmp_size->buffer_bytes;

	/* Now create the shared memory segment */
	shmid = shmget(SHMKEY, cd_tmp_size->size,
		       IPC_PERMS|IPC_CREAT|IPC_EXCL);
	if (shmid < 0) {
		/*
		 * If shared memory exists already, load it, remove it
		 * and create new one. This should not be necessary
		 * if the application previously exited correctly.
		 */
		if (errno == EEXIST) {
			shmid = shmget(SHMKEY, cd_tmp_size->size,
				       IPC_PERMS|IPC_EXCL);
			if (shmid < 0) {
				(void) sprintf(errbuf,
					    "cdda_getshm: shmget failed "
					    "(errno=%d)",
					    errno);
				CDDA_INFO(errbuf);
				DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
				return FALSE;
			}

			if (shmctl(shmid, IPC_RMID, NULL) != 0) {
				(void) sprintf(errbuf,
					    "cdda_getshm: shmctl IPC_RMID "
					    "failed (errno=%d)",
					    errno);
				CDDA_INFO(errbuf);
				DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
				return FALSE;
			}

			shmid = shmget(SHMKEY, cd_tmp_size->size,
				       IPC_PERMS|IPC_CREAT|IPC_EXCL);
			if (shmid < 0) {
				(void) sprintf(errbuf,
					    "cdda_getshm: shmget failed "
					    "(errno=%d)",
					    errno);
				CDDA_INFO(errbuf);
				DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
				return FALSE;
			}
		}
		else {
			(void) sprintf(errbuf,
				    "cdda_getshm: shmget failed (errno=%d)",
				    errno);
			CDDA_INFO(errbuf);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
			return FALSE;
		}
	}

	/* Attach */
	if ((cdda_shmaddr = (void *) shmat(shmid, NULL, 0)) == NULL) {
		(void) sprintf(errbuf,
				"cdda_getshm: shmat failed (errno=%d)",
				errno);
		CDDA_INFO(errbuf);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		return FALSE;
	}

	/* Now set our fields pointing into the shared memory */
	cd->cds = (cd_size_t *) cdda_shmaddr;

	/* Copy in sizes */
	(void) memcpy(cdda_shmaddr, cd_tmp_size, sizeof(cd_size_t));

	/* Free temporary cd_size_t */
	MEM_FREE(cd_tmp_size);

	/* Info */
	cd->i = (cd_info_t *)(void *)
		((byte_t *) cdda_shmaddr + sizeof(cd_size_t));

	/* Buffer */
	cd->cdb = (cd_buffer_t *)(void *)
		((byte_t *) cd->i + sizeof(cd_info_t));

	/* Overlap */
	cd->cdb->olap = (byte_t *) ((byte_t *) cd->cdb + sizeof(cd_buffer_t));

	/* Data */
	cd->cdb->data = (byte_t *)
			((byte_t *) cd->cdb->olap +
				    (cd->cds->search_bytes << 1));

	/* Buffer */
	cd->cdb->b = (byte_t *)
		     ((byte_t *) cd->cdb->data +
				 cd->cds->chunk_bytes +
				 (cd->cds->olap_bytes << 1));

	return TRUE;
}


/*
 * cdda_initshm
 *	Initialize fields of the cd_state_t structure. The cd_size_t structure
 *	has already been done in cdda_getshm(). The buffer we initialize
 *	prior to playing.
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
cdda_initshm(curstat_t *s)
{
	/* General state */
	cd->i->state = CDSTAT_COMPLETED;
	cd->i->chroute = app_data.ch_route;
	cd->i->vol_taper = app_data.vol_taper;
	cd->i->vol = 0;
	cd->i->vol_left = 100;
	cd->i->vol_right = 100;
	cd->i->jitter = (int) app_data.cdda_jitter_corr;
	cd->i->frm_played = 0;
	cd->i->trk_idx = 0;
	cd->i->trk_len = 0;
	cd->i->debug = app_data.debug;

	/* Start and ending LBAs for tracks */
	cd->i->start_lba = 0;
	cd->i->end_lba = 0;

	/* Process ids */
	cd->i->reader = 0;
	cd->i->writer = 0;
}


/*
 * cdda_sysvipc_cleanup
 *	Detaches shared memory, removes it and destroys semaphores.
 *
 * Args:
 *	fd - Device file descriptor
 *	s  - Pointer to the curstat_t structure
 *
 * Return:
 *	Nothing.
 */
STATIC void
cdda_sysvipc_cleanup(void)
{
	DBGPRN(DBG_DEVIO)(errfp,
			  "\ncdda_sysvipc_cleanup: Cleaning up CDDA.\n");

	/* Detach */
	if (cd->cds != NULL) {
		if (shmdt((char *) cd->cds) < 0) {
			(void) sprintf(errbuf,
			    "cdda_sysvipc_cleanup: shmdt failed (errno=%d)",
			    errno);
			CDDA_INFO(errbuf);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		}
		else
			cd->cds = NULL;
	}

	/* Remove shared memory */
	if (shmid >= 0) {
		if (shmctl(shmid, IPC_RMID, NULL) < 0) {
			(void) sprintf(errbuf,
			    "cdda_sysvipc_cleanup: shmctl IPC_RMID failed "
			    "(errno=%d)",
			    errno);
			CDDA_INFO(errbuf);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		}
		else
			shmid = -1;
	}

	/* Destroy semaphores */
	if (semid >= 0) {
		semun_t	arg;

		arg.val = 0;
		if (semctl(semid, 0, IPC_RMID, arg) < 0) {
			(void) sprintf(errbuf,
			    "cdda_sysvipc_cleanup: semctl IPC_RMID failed "
			    "(errno=%d)",
			    errno);
			CDDA_INFO(errbuf);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		}
		else
			semid = -1;
	}
}


/*
 * cdda_sysvipc_capab
 *	Return configured CDDA capabilities
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported CDDA read and write features
 */
word32_t
cdda_sysvipc_capab(void)
{
	word32_t	(*rinit)(void),
			(*winit)(void);

	if (app_data.cdda_rdmethod == CDDA_RD_NONE ||
	    app_data.cdda_wrmethod == CDDA_WR_NONE)
		return 0;

	if (app_data.cdda_rdmethod < CDDA_RD_NONE ||
	    app_data.cdda_rdmethod >= CDDA_RD_METHODS ||
	    app_data.cdda_wrmethod < CDDA_WR_NONE ||
	    app_data.cdda_wrmethod >= CDDA_WR_METHODS) {
		CDDA_WARNING(app_data.str_cddainit_fail);
		DBGPRN(DBG_DEVIO)(errfp, "Warning: %s\n",
				  app_data.str_cddainit_fail);
		return 0;
	}

	/* Call readinit and writeinit functions to determine capability */
	rinit = sysvipc_rd_calltbl[app_data.cdda_rdmethod].readinit;
	winit = sysvipc_wr_calltbl[app_data.cdda_wrmethod].writeinit;

	if (rinit == NULL || winit == NULL) {
		CDDA_WARNING(app_data.str_cddainit_fail);
		DBGPRN(DBG_DEVIO)(errfp, "Warning: %s\n",
				  app_data.str_cddainit_fail);
		return 0;
	}

	return ((*rinit)() | (*winit)());
}


/*
 * cdda_sysvipc_init
 *	Initialize cdda subsystem.
 *
 * Args:
 *	s - Pointer to the curstat_t structure.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
cdda_sysvipc_init(curstat_t *s)
{
	if (app_data.cdda_rdmethod <= CDDA_RD_NONE ||
	    app_data.cdda_rdmethod >= CDDA_RD_METHODS ||
	    app_data.cdda_wrmethod <= CDDA_WR_NONE ||
	    app_data.cdda_wrmethod >= CDDA_WR_METHODS ||
	    sysvipc_rd_calltbl[app_data.cdda_rdmethod].read == NULL ||
	    sysvipc_wr_calltbl[app_data.cdda_wrmethod].write == NULL) {
		return FALSE;
	}

	/* Allocate memory */
	cd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (cd == NULL)
		/* Out of memory */
		return FALSE;

	(void) memset(cd, 0, sizeof(cd_state_t));

	/* Initialize semaphores */
	if (!cdda_getsem()) {
		cdda_sysvipc_cleanup();
		return FALSE;
	}

	/* Initialize shared memory */
	if (!cdda_getshm()) {
		cdda_sysvipc_cleanup();
		return FALSE;
	}

	/* Initialize fields */
	cdda_initshm(s);

	DBGPRN(DBG_DEVIO)(errfp, "\ncdda_sysvipc_init: CDDA initted.\n");
	return TRUE;
}


/*
 * cdda_sysvipc_halt
 *	Halt cdda subsystem.
 *
 * Args:
 *	fd - Device file descriptor
 *	s -  Pointer to the curstat_t structure
 *
 * Return:
 *	Nothing.
 */
void
cdda_sysvipc_halt(int fd, curstat_t *s)
{
	/* Stop playback, if applicable */
	(void) cdda_sysvipc_stop(fd, s);

	/* Clean up IPC */
	cdda_sysvipc_cleanup();

	DBGPRN(DBG_DEVIO)(errfp, "\ncdda_sysvipc_halt: CDDA halted.\n");
}


/*
 * cdda_sysvipc_play
 *	Start cdda play.
 *
 * Args:
 *	fd - Device file descriptor
 *	s -  Pointer to the curstat_t structure
 *	start_lba - Start logical block address
 *	end_lba   - End logical block address
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
bool_t
cdda_sysvipc_play(int fd, curstat_t *s, int start_lba, int end_lba)
{
	int	i;
	pid_t	cpid;
	semun_t	arg;
	bool_t	(*readf)(int),
		(*writef)(byte_t, byte_t, curstat_t *);

	readf = sysvipc_rd_calltbl[app_data.cdda_rdmethod].read;
	writef = sysvipc_wr_calltbl[app_data.cdda_wrmethod].write;

	/* If not stopped, stop first */
	if (cd->i->state != CDSTAT_COMPLETED)
		(void) cdda_sysvipc_stop(fd, s);

	/* Set status */
	cd->i->state = CDSTAT_PLAYING;

	/* Where are we starting */
	cd->i->start_lba = start_lba;
	cd->i->end_lba = end_lba;

	/* Not finished reading */
	cd->i->cdda_done = 0;

	/* Buffer pointers */
	cd->cdb->occupied = 0;
	cd->cdb->nextin = 0;
	cd->cdb->nextout = 0;

	/* Clear message buffer */
	cd->i->msgbuf[0] = '\0';

	/* Room available */
	arg.val = 1;
	if (semctl(semid, ROOM, SETVAL, arg) < 0) {
		(void) sprintf(errbuf,
			    "cdda_sysvipc_play: semctl SETVAL failed "
			    "(errno=%d)",
			    errno);
		CDDA_INFO(errbuf);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		cd->i->state = CDSTAT_COMPLETED;
		return FALSE;
	}

	/* No data available */
	arg.val = 0;
	if (semctl(semid, DATA, SETVAL, arg) < 0) {
		(void) sprintf(errbuf,
			    "cdda_sysvipc_play: semctl SETVAL failed "
			    "(errno=%d)",
			    errno);
		CDDA_INFO(errbuf);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		cd->i->state = CDSTAT_COMPLETED;
		return FALSE;
	}

	/* Fork CDDA reader process */
	switch (cpid = FORK()) {
	case -1:
		(void) sprintf(errbuf,
			    "cdda_sysvipc_play: fork failed (reader) "
			    "(errno=%d)",
			    errno);
		CDDA_INFO(errbuf);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		cd->i->state = CDSTAT_COMPLETED;
		if (cd->i->writer > 0) {
			(void) kill(cd->i->writer, SIGTERM);
			cd->i->writer = 0;
		}
		return FALSE;

	case 0:
		/* Child */

		/* Close any unneeded file descriptors */
		for (i = 3; i < 10; i++) {
			if (i != fd)
				(void) close(i);
		}

		/* Call read method */
		if (!(*readf)(fd))
			exit(1);

		exit(0);
		/*NOTREACHED*/

	default:
		/* Parent */
		DBGPRN(DBG_DEVIO)(errfp,
				"\ncdda_sysvipc_play: "
				"Started reader process pid=%d\n",
				(int) cpid);
		break;
	}

	/* Store reader pid */
	cd->i->reader = cpid;

	/* Fork CDDA writer process */
	switch (cpid = FORK()) {
	case -1:
		(void) sprintf(errbuf,
			    "cdda_sysvipc_play: fork failed (writer) "
			    "(errno=%d)",
			    errno);
		CDDA_INFO(errbuf);
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
		cd->i->state = CDSTAT_COMPLETED;
		if (cd->i->reader > 0) {
			(void) kill(cd->i->reader, SIGTERM);
			cd->i->reader = 0;
		}
		return FALSE;

	case 0:
		/* Child */

		/* Close any unneeded file descriptors */
		for (i = 3; i < 10; i++)
			(void) close(i);

#ifndef HAS_SETEUID
		/* Set to original uid/gid */
		util_set_ougid();
#endif

		/* Call write method */
		if (!(*writef)(app_data.play_mode, app_data.cdda_filefmt, s))
			exit(1);

		exit(0);
		/*NOTREACHED*/

	default:
		/* Parent */
		DBGPRN(DBG_DEVIO)(errfp,
				"\ncdda_sysvipc_play: "
				"Started writer process pid=%d\n",
				(int) cpid);
		break;
	}

	/* Store writer pid */
	cd->i->writer = cpid;

	return TRUE;
}


/*
 * cdda_sysvipc_pause_resume
 *	Pause CDDA playback.
 *
 * Args:
 *	fd     - Device file descriptor
 *	s      -  Pointer to the curstat_t structure
 *	resume - Whether to resume playback
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
bool_t
cdda_sysvipc_pause_resume(int fd, curstat_t *s, bool_t resume)
{
	if (cd->i->writer > 0) {
		if (resume)
			cd->i->state = CDSTAT_PLAYING;
		else
			cd->i->state = CDSTAT_PAUSED;

		/* Signal writer if active */
		if (kill(cd->i->writer, SIGUSR1) < 0) {
			(void) sprintf(errbuf,
				    "cdda_sysvipc_pause_resume: Cannot send "
				    "SIGUSR1 to writer (errno=%d)",
				    errno);
			CDDA_INFO(errbuf);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
			return FALSE;
		}
	}
	return TRUE;
}


/*
 * cdda_sysvipc_stop
 *	Stop CDDA playback.
 *
 * Args:
 *	fd - Device file descriptor
 *	s -  Pointer to the curstat_t structure
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
bool_t
cdda_sysvipc_stop(int fd, curstat_t *s)
{
	int	ret,
		wstat;
	pid_t	cpid;

	/* Set status */
	cd->i->state = CDSTAT_COMPLETED;

	if (cd->i->writer != 0) {
		cpid = cd->i->writer;

		/* Wait for writer, blocking */
		DBGPRN(DBG_DEVIO)(errfp,
			"\ncdda_sysvipc_stop: Waiting for writer pid=%d\n",
			(int) cpid);

		while ((ret = WAITPID(cpid, &wstat, 0)) != cpid) {
			if (ret < 0 && errno != EINTR)
				break;
		}

		if (WIFEXITED(wstat)) {
			ret = WEXITSTATUS(wstat);
			DBGPRN(DBG_DEVIO)(errfp, "Writer exited, status %d\n",
					  ret);

			/* Display message, if any */
			if (ret != 0 && cd->i->msgbuf[0] != '\0')
				CDDA_INFO(cd->i->msgbuf);
		}
		else if (WIFSIGNALED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp, "Writer killed, signal %d\n",
					  WTERMSIG(wstat));
		}

		cd->i->writer = 0;
	}

	if (cd->i->reader != 0) {
		cpid = cd->i->reader;

		/* Wait for reader, blocking */
		DBGPRN(DBG_DEVIO)(errfp,
			"\ncdda_sysvipc_stop: Waiting for reader pid=%d\n",
			(int) cd->i->reader);

		while ((ret = WAITPID(cpid, &wstat, 0)) != cpid) {
			if (ret < 0 && errno != EINTR)
				break;
		}

		if (WIFEXITED(wstat)) {
			ret = WEXITSTATUS(wstat);
			DBGPRN(DBG_DEVIO)(errfp, "Reader exited, status %d\n",
					  ret);

			/* Display message, if any */
			if (ret != 0 && cd->i->msgbuf[0] != '\0')
				CDDA_INFO(cd->i->msgbuf);
		}
		else if (WIFSIGNALED(wstat)) {
			DBGPRN(DBG_DEVIO)(errfp, "Reader killed, signal %d\n",
					  WTERMSIG(wstat));
		}

		cd->i->reader = 0;
	}

	return TRUE;
}


/*
 * cdda_sysvipc_vol
 *	Change volume setting.
 *
 * Args:
 *	fd    - Device file descriptor
 *	s     - Pointer to the curstat_t structure
 *	vol   - Desired volume level
 *	query - Whether querying or setting the volume
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
int
cdda_sysvipc_vol(int fd, curstat_t *s, int vol, bool_t query)
{
	if (query) {
		s->level_left = (byte_t) cd->i->vol_left;
		s->level_right = (byte_t) cd->i->vol_right;
		return (cd->i->vol);
	}

	cd->i->vol_taper = app_data.vol_taper;
	cd->i->vol = vol;
	cd->i->vol_left = (int) s->level_left;
	cd->i->vol_right = (int) s->level_right;

	/* Signal writer if active */
	if (cd->i->writer > 0) {
		if (kill(cd->i->writer, SIGUSR1) < 0) {
			(void) sprintf(errbuf,
				    "cdda_sysvipc_vol: Cannot send "
				    "SIGUSR1 to writer (errno=%d)",
				    errno);
			CDDA_INFO(errbuf);
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", errbuf);
			return -1;
		}
	}

	return (vol);
}


/*
 * cdda_sysvipc_chroute
 *	Change channel routing setting.
 *
 * Args:
 *	fd - Device file descriptor
 *	s -  Pointer to the curstat_t structure
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
bool_t
cdda_sysvipc_chroute(int fd, curstat_t *s)
{
	cd->i->chroute = app_data.ch_route;
	return TRUE;
}


/*
 * cdda_sysvipc_getstatus
 *	Get CDDA playback status.
 *
 * Args:
 *	fd - Device file descriptor
 *	s -  Pointer to curstat_t structure
 *	sp - CDDA status return structure
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
bool_t
cdda_sysvipc_getstatus(int fd, curstat_t *s, cdstat_t *sp)
{
	int	i;

	/* Current playback status */
	sp->status = cd->i->state;
	if (sp->status == CDSTAT_COMPLETED)
		(void) cdda_sysvipc_stop(fd, s);

	/* Current playback location */
	sp->abs_addr.addr = (word32_t) (cd->i->start_lba + cd->i->frm_played);

	util_blktomsf(
		sp->abs_addr.addr,
		&sp->abs_addr.min,
		&sp->abs_addr.sec,
		&sp->abs_addr.frame,
		MSF_OFFSET
	);

	i = cd->i->trk_idx;
	if (sp->abs_addr.addr >= s->trkinfo[i].addr &&
	    sp->abs_addr.addr < s->trkinfo[i+1].addr) {
			sp->track = s->trkinfo[i].trkno;

			sp->rel_addr.addr =
				(word32_t) sp->abs_addr.addr -
				s->trkinfo[i].addr;

			util_blktomsf(
				sp->rel_addr.addr,
				&sp->rel_addr.min,
				&sp->rel_addr.sec,
				&sp->rel_addr.frame,
				0
			);
	}
	else for (i = 0; i < (int) s->tot_trks; i++) {
		if (sp->abs_addr.addr >= s->trkinfo[i].addr &&
		    sp->abs_addr.addr < s->trkinfo[i+1].addr) {
			sp->track = s->trkinfo[i].trkno;

			sp->rel_addr.addr =
				(word32_t) sp->abs_addr.addr -
				s->trkinfo[i].addr;

			util_blktomsf(
				sp->rel_addr.addr,
				&sp->rel_addr.min,
				&sp->rel_addr.sec,
				&sp->rel_addr.frame,
				0
			);
			break;
		}
	}

	sp->index = 1;	/* Index number not supported in this mode */

	/* Current volume and balance */
	sp->level = (byte_t) cd->i->vol;
	sp->level_left = (byte_t) cd->i->vol_left;
	sp->level_right = (byte_t) cd->i->vol_right;

	return TRUE;
}


/*
 * cdda_sysvipc_debug
 *	Debug level change notification function
 *
 * Args:
 *	lev - New debug level
 *
 * Return:
 *	Nothing.
 */
void
cdda_sysvipc_debug(word32_t lev)
{
	cd->i->debug = lev;
}


/*
 * cdda_rw_initipc
 *	Retrieves shared memory and semaphores. Sets up our pointers
 *	from cd into shared memory.  Used by the reader/writer child
 *	processes.
 *
 * Args:
 *	cd - Pointer to the cd_state_t structure to be filled in
 *
 * Return:
 *	The IPC semaphore ID, or -1 if failed.
 */
int
cdda_rw_initipc(cd_state_t *cdp)
{
	int	id;

	if (cdda_shmaddr == NULL) {
		DBGPRN(DBG_DEVIO)(errfp,
			"gen_write_initipc: No shared memory!\n");
		return -1;
	}

	/* Now set our fields pointing into the shared memory */
	cdp->cds = (cd_size_t *) cdda_shmaddr;

	/* Info */
	cdp->i = (cd_info_t *)(void *)
		((byte_t *) cdda_shmaddr + sizeof(cd_size_t));

	/* Buffer */
	cdp->cdb = (cd_buffer_t *)(void *)
		((byte_t *) cd->i + sizeof(cd_info_t));

	/* Overlap */
	cdp->cdb->olap = (byte_t *) ((byte_t *) cd->cdb + sizeof(cd_buffer_t));

	/* Data */
	cdp->cdb->data = (byte_t *)
			 ((byte_t *) cd->cdb->olap +
				    (cd->cds->search_bytes << 1));

	/* Buffer */
	cdp->cdb->b = (byte_t *)
		      ((byte_t *) cd->cdb->data + cd->cds->chunk_bytes +
				 (cd->cds->olap_bytes << 1));

	/* Semaphores */
	if ((id = semget(SEMKEY, 3, IPC_PERMS|IPC_EXCL)) < 0) {
		DBGPRN(DBG_DEVIO)(errfp,
			"gen_write_initipc: semget failed: errno=%d\n",
			errno);
	}

	return (id);
}


/*
 * cdda_rd_corrjitter
 *	Correct for jitter. The olap array contains cdp->cds->search_bytes * 2
 *	of data and the last data written to the device is situated at
 *	olap[cdp->cds->search_bytes]. We try to match the last data read,
 *	which starts at data[cdp->cds->olap_bytes], around the center of olap.
 *	Once matched we may have to copy some data from olap into data
 *	(overshoot) or skip some of the audio in data (undershoot). The
 *	function returns a pointer into data indicating from where we should
 *	start writing.
 *	Used by the reader child process only.
 *
 * Args:
 *	None.
 *
 * Return:
 *	Pointer into data
 */
byte_t *
cdda_rd_corrjitter(cd_state_t *cdp)
{
	int		idx,
			d;
	byte_t		*i,
			*j,
			*l;

	/* If jitter correction off, return */
	if (!cdp->i->jitter)
		return (cdp->cdb->data);

	/* Find a match in olap, if we can */
	i = cdp->cdb->olap;
	l = &cdp->cdb->olap[(cdp->cds->search_bytes << 1) - 1];
	j = &cdp->cdb->data[cdp->cds->olap_bytes];

	do {
		i = memchr(i, (int) *j, l - i);
		if (i != NULL) {
			if ((i - cdp->cdb->olap) % 2 == 0) {
				d = memcmp(j, i, cdp->cds->str_length);
				if (d == 0)
					break;
			}
			i++;
		}
	} while (i != NULL && i < l);

	/* Did we find a match? */
	if (i == NULL || i >= l) {
		DBGPRN(DBG_DEVIO)(errfp,
		    "cdda_rd_corrjitter: Could not correct for jitter.\n");
		idx = 0;
	}
	else {
		idx = i - cdp->cdb->olap - cdp->cds->search_bytes;
	}

	/* Update pointer */
	j -= idx;

	/* Match on RHS, copy some olap into data */
	if (idx > 0)
		(void) memcpy(j, cdp->cdb->olap + cdp->cds->search_bytes, idx);

	return (j);
}


/*
 * cdda_rd_setcurtrk
 *	Set the current playing track index based on the lba.
 *
 * Args:
 *	cdp - Pointer to the cd_state_t structure
 *	lba - The current playing logical block address
 *
 * Return:
 *	Nothing.
 */
void
cdda_rd_setcurtrk(cd_state_t *cdp, int lba)
{
	int		i;
	curstat_t	*s = cdda_clinfo->curstat_addr();

	if (cdp->i->state != CDSTAT_PLAYING && cdp->i->state != CDSTAT_PAUSED)
		return;

	i = cdp->i->trk_idx;
	if (i >= 0 &&
	    lba >= s->trkinfo[i].addr && lba < s->trkinfo[i+1].addr) {
		cdp->i->trk_idx = i;
		cdp->i->trk_len = CDDA_BLKSZ *
			(s->trkinfo[i+1].addr - s->trkinfo[i].addr + 1);
		return;
	}

	for (i = 0; i < (int) s->tot_trks; i++) {
		if (lba >= s->trkinfo[i].addr &&
		    lba < s->trkinfo[i+1].addr) {
			cdp->i->trk_idx = i;
			cdp->i->trk_len = CDDA_BLKSZ *
			    (s->trkinfo[i+1].addr - s->trkinfo[i].addr + 1);
		}
	}
}


/*
 * cdda_wr_getcurtrk
 *	Determine the current playing track index based on the lba.
 *
 * Args:
 *	cdp - Pointer to the cd_state_t structure
 *	lba - The current playing logical block address
 *
 * Return:
 *	The track index number, or -1 if failed
 */
int
cdda_wr_getcurtrk(cd_state_t *cdp, int lba)
{
	int		i;
	curstat_t	*s = cdda_clinfo->curstat_addr();

	if (cdp->i->state != CDSTAT_PLAYING && cdp->i->state != CDSTAT_PAUSED)
		return -1;

	i = cdp->i->trk_idx;
	if (i >= 0 &&
	    lba >= s->trkinfo[i].addr && lba < s->trkinfo[i + 1].addr)
		return (i);

	for (i = 0; i < (int) s->tot_trks; i++) {
		if (lba >= s->trkinfo[i].addr && lba < s->trkinfo[i + 1].addr)
			return (i);
	}

	return -1;
}


#endif	/* CDDA_SYSVIPC */

