/*
 *   libdi - scsipt SCSI Device Interface Library
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   Silicon Graphics Irix support
 *
 *   This software fragment contains code that interfaces the
 *   application to the SGI Irix operating system.  The names "SGI" and
 *   "Irix" are used here for identification purposes only.
 */
#ifndef LINT
static char *_os_irix_c_ident_ = "@(#)os_irix.c	6.36 02/03/17";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "libdi_d/libdi.h"
#include "libdi_d/scsipt.h"

#if defined(sgi) && defined(DI_SCSIPT) && !defined(DEMO_ONLY)

extern appdata_t	app_data;
extern bool_t		scsipt_notrom_error;
extern FILE		*errfp;
extern di_client_t	*di_clinfo;

STATIC req_sense_data_t	sense_data;	/* Request sense data buffer */


/*
 * pthru_send
 *	Send SCSI command to the device.
 *
 * Args:
 *	fd - Device file descriptor
 *	cmdpt - Pointer to the SCSI command CDB
 *	cmdlen - SCSI command size (6, 10 or 12 bytes)
 *	datapt - Pointer to the data buffer
 *	datalen - Data transfer size (bytes)
 *	rw - Data transfer direction flag (OP_NODATA, OP_DATAIN or OP_DATAOUT)
 *	tmout - Command timeout interval (seconds)
 *	prnerr - Whether an error message should be displayed
 *		 when a command fails
 *
 * Return:
 *	TRUE - command completed successfully
 *	FALSE - command failed
 */
bool_t
pthru_send(
	int		fd,
	byte_t		*cmdpt,
	int		cmdlen,
	byte_t		*datapt,
	int		datalen,
	byte_t		rw,
	int		tmout,
	bool_t		prnerr
)
{
	struct dsreq	dsreq;
	char		*path,
			title[FILE_PATH_SZ + 20];

	if (fd < 0 || scsipt_notrom_error)
		return FALSE;

	path = di_devgetpath(fd);

	(void) memset(&dsreq, 0, sizeof(dsreq));
	(void) memset(&sense_data, 0, sizeof(sense_data));

	/* Set up dsreq */
	dsreq.ds_cmdbuf = (caddr_t) cmdpt;
	dsreq.ds_cmdlen = cmdlen;
	if (datapt != NULL && datalen != 0) {
		switch (rw) {
		case OP_DATAIN:
			dsreq.ds_flags |= DSRQ_READ;
			break;
		case OP_DATAOUT:
			dsreq.ds_flags |= DSRQ_WRITE;
			break;
		case OP_NODATA:
		default:
			break;
		}
		dsreq.ds_databuf = (caddr_t) datapt;
		dsreq.ds_datalen = (ulong) datalen;
	}
	dsreq.ds_flags |= DSRQ_SENSE;
	dsreq.ds_sensebuf = (caddr_t) &sense_data;
	dsreq.ds_senselen = (ulong) SZ_RSENSE;
	dsreq.ds_time = (tmout ? tmout : DFLT_CMD_TIMEOUT) * 1000;

	if (app_data.debug & DBG_DEVIO) {
		(void) sprintf(title, "SCSI CDB bytes (dev=%s rw=%d to=%d)",
			       path, rw, tmout);
		util_dbgdump(title, cmdpt, cmdlen);
	}

	/* Send the command down via the "pass-through" interface */
	if (ioctl(fd, DS_ENTER, &dsreq) < 0) {
		if (app_data.scsierr_msg && prnerr) 
			perror("DS_ENTER ioctl failed");
		return FALSE;
	}

	if (dsreq.ds_ret != 0) {
		if (app_data.scsierr_msg && prnerr) {
			(void) fprintf(errfp,
				    "%s: %s %s:\n%s=0x%x %s=0x%x %s=0x%x",
				    APPNAME,
				    "SCSI command error on",
				    path,
				    "Opcode",
				    cmdpt[0],
				    "Ret",
				    dsreq.ds_ret,
				    "Status",
				    dsreq.ds_status);

			if (sense_data.key == 0)
				(void) fprintf(errfp, "\n");
			else {
				(void) fprintf(errfp,
					    " Key=0x%x Code=0x%x Qual=0x%x\n",
					    sense_data.key,
					    sense_data.code,
					    sense_data.qual);
			}
		}

		return FALSE;
	}
	
	return TRUE;
}


/*
 * pthru_open
 *	Open SCSI pass-through device
 *
 * Args:
 *	path - device path name string
 *
 * Return:
 *	Device file descriptor, or -1 on failure.
 */
int
pthru_open(char *path)
{
	int		fd;
	struct stat	stbuf;
	char		errstr[ERR_BUF_SZ];

	/* Check for validity of device node */
	if (stat(path, &stbuf) < 0) {
		(void) sprintf(errstr, app_data.str_staterr, path);
		DI_FATAL(errstr);
		return -1;
	}

	if (!S_ISCHR(stbuf.st_mode)) {
		(void) sprintf(errstr, app_data.str_noderr, path);
		DI_FATAL(errstr);
		return -1;
	}

	if ((fd = open(path, O_RDONLY)) < 0) {
		DBGPRN(DBG_DEVIO)(errfp,
			"Cannot open %s: errno=%d\n", path, errno);
		return -1;
	}

	(void) di_devreg(fd, path, 0);

	/* SGI hack:  The CD-ROM driver allows the open to succeed
	 * even if there is no CD loaded.  We test for the existence of
	 * a disc with scsipt_disc_present().
	 */
	if (!scsipt_disc_present(fd)) {
		/* No CD loaded */
		pthru_close(fd);
		return -1;
	}

	return (fd);
}


/*
 * pthru_close
 *	Close SCSI pass-through device
 *
 * Args:
 *	fd - Device file descriptor
 *
 * Return:
 *	Nothing.
 */
void
pthru_close(int fd)
{
	di_devunreg(fd);
	(void) close(fd);
}


/*
 * pthru_vers
 *	Return OS Interface Module version string
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Module version text string.
 */
char *
pthru_vers(void)
{
	return ("OS Interface module for SGI Irix");
}

#endif	/* sgi DI_SCSIPT DEMO_ONLY */

