/*
 *   libdi - scsipt SCSI Device Interface Library
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   Linux support
 *
 *   This software fragment contains code that interfaces the
 *   application to the Linux operating system.
 */
#ifndef LINT
static char *_os_linux_c_ident_ = "@(#)os_linux.c	6.46 02/04/07";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "libdi_d/libdi.h"
#include "libdi_d/scsipt.h"

#if defined(linux) && defined(DI_SCSIPT) && !defined(DEMO_ONLY)

extern appdata_t	app_data;
extern bool_t		scsipt_notrom_error;
extern FILE		*errfp;
extern di_client_t	*di_clinfo;

STATIC sg_request_t	req,
			rep;
STATIC int		prev_tmout = -1;


/*
 * find_sg
 *	Given a minor device number of a non-sg CD device, locate the
 *	associated sg device and return its index number.  This
 *	function reads /proc/scsi/scsi to obtain the information.
 *	Note: This function makes assumptions about the output format of
 *	the output of /proc/scsi/scsi.
 *
 * Args:
 *	path - The non-sg device path
 *	mdev - The non-sg device minor device number
 *
 * Return:
 *	The sg device index, or -1 if none is found.
 */
STATIC int
find_sg(char *path, int mdev)
{
	FILE		*fp;
	int		i,
			j;
	char		typestr[STR_BUF_SZ * 2],
			dummy[STR_BUF_SZ * 2],
			buf[STR_BUF_SZ * 2];

	if ((fp = fopen(PROC_SCSI_SCSI, "r")) == NULL)
		return -1;

	i = j = -1;
	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if (sscanf(buf, " Type: %s %[^\n]\n", typestr, dummy) == 0)
			continue;

		i++;
		if (strcmp(typestr, "CD-ROM") == 0 ||
		    strcmp(typestr, "WORM") == 0) {
			if (++j == mdev)
				break;
		}
	}
	(void) fclose(fp);

	return (i);
}


/*
 * pthru_send
 *	Send SCSI command to the device.
 *
 * Args:
 *	fd - Device file descriptor
 *	cmdpt - Pointer to the SCSI command CDB
 *	cmdlen - SCSI command size (6, 10 or 12 bytes)
 *	datapt - Pointer to the data buffer
 *	datalen - Data transfer size (bytes)
 *	rw - Data transfer direction flag (OP_NODATA, OP_DATAIN or OP_DATAOUT)
 *	tmout - Command timeout interval (seconds)
 *	prnerr - Whether an error message should be displayed
 *		 when a command fails
 *
 * Return:
 *	TRUE - command completed successfully
 *	FALSE - command failed
 */
bool_t
pthru_send(
	int		fd,
	byte_t		*cmdpt,
	int		cmdlen,
	byte_t		*datapt,
	int		datalen,
	byte_t		rw,
	int		tmout,
	bool_t		prnerr
)
{
	byte_t			*ptbuf;
	int			reqsz,
				ptbufsz,
				ret;
	req_sense_data_t	*rp;
	di_devreg_t		*regp;
	char			*path,
				title[FILE_PATH_SZ + 20];
	void			(*oint)(int),
				(*oquit)(int),
				(*opipe)(int),
				(*oterm)(int);

	if (fd < 0 || scsipt_notrom_error)
		return FALSE;

	if ((regp = di_devgetent(fd)) == NULL)
		return FALSE;

	if (app_data.debug & DBG_DEVIO) {
		(void) sprintf(title, "SCSI CDB bytes (dev=%s rw=%d to=%d)",
				regp->path, rw, tmout);
		util_dbgdump(title, cmdpt, cmdlen);
	}

	/* Hack: use SCSI_IOCTL_TEST_UNIT_READY ioctl for test unit ready
	 * commands.  The SCSI Generic interface would return success even
	 * when the device is actually not ready (such as when the medium is
	 * not loaded).
	 */
	if (cmdpt[0] == OP_S_TEST) {
		DBGPRN(DBG_DEVIO)(errfp,
			"\nSending SCSI_IOCTL_TEST_UNIT_READY (%s)\n",
			regp->path);

		ret = ioctl(fd, SCSI_IOCTL_TEST_UNIT_READY, NULL);
		if (ret == 0)
			return TRUE;

		if (app_data.scsierr_msg && prnerr) {
			(void) fprintf(errfp,
				"SCSI_IOCTL_TEST_UNIT_READY failed: "
				"ret=0x%x\n",
				ret
			);
		}
		return FALSE;
	}

	if ((regp->flags & 0x1) == 0) {
		/* Use SCSI ioctl interface */

		/* Set up SCSI pass-through command/data buffer */
		ptbufsz = (sizeof(int) << 1) + cmdlen +
			  sizeof(req_sense_data_t);
		if (datapt != NULL && datalen > 0)
			ptbufsz += datalen;

		if ((ptbuf = (byte_t *) MEM_ALLOC("ptbuf", ptbufsz)) == NULL) {
			DI_FATAL(app_data.str_nomemory);
			return FALSE;
		}

		(void) memset(ptbuf, 0, ptbufsz);
		(void) memcpy(ptbuf + (sizeof(int) << 1), cmdpt, cmdlen);

		if (datapt != NULL && datalen > 0) {
			if (ptbufsz > SCSI_IOCTL_MAX_BUF) {
				DBGPRN(DBG_DEVIO)(errfp,
					"SCSI command failure: %s\n",
					"I/O data size too large");
				MEM_FREE(ptbuf);
				return FALSE;
			}

			if (rw == OP_DATAOUT) {
				*((int *) ptbuf) = datalen;
				(void) memcpy(
					ptbuf + (sizeof(int) << 1) + cmdlen,
					datapt,
					datalen
				);
			}
			else
				*(((int *) ptbuf) + 1) = datalen;
		}

		/* Send command */
		if ((ret = ioctl(fd, SCSI_IOCTL_SEND_COMMAND, ptbuf)) != 0) {
			if (app_data.scsierr_msg && prnerr) {
				(void) fprintf(errfp, "%s: %s %s:\n",
					APPNAME,
					"SCSI command error on", regp->path);
				(void) fprintf(errfp,
					"%s=0x%x %s=0x%x %s=0x%x "
					"%s=0x%x %s=0x%x\n",
					"Opcode", cmdpt[0],
					"Status", status_byte(ret),
					"Msg", msg_byte(ret),
					"Host", host_byte(ret),
					"Driver", driver_byte(ret));

				rp = (req_sense_data_t *)(void *)
					(ptbuf + (sizeof(int) << 1));

				if (rp->key != 0) {
					(void) fprintf(errfp,
					    "Key=0x%x Code=0x%x Qual=0x%x\n",
					    rp->key, rp->code, rp->qual);
				}
			}

			MEM_FREE(ptbuf);
			return FALSE;
		}

		if (datapt != NULL && rw == OP_DATAIN && datalen > 0)
			(void) memcpy(datapt, ptbuf + (sizeof(int) << 1),
				      datalen);

		MEM_FREE(ptbuf);
	}
	else {
		/* Use SCSI Generic interface */

		/* Set command timeout interval if needed */
		if (tmout != prev_tmout) {
			unsigned long	jiffies;

			prev_tmout = tmout;
			jiffies = (unsigned long) tmout * HZ;

			if (ioctl(fd, SG_SET_TIMEOUT, &jiffies) < 0) {
				(void) fprintf(errfp,
					"SCSI command failure: %s: %s\n",
					"SG_SET_TIMEOUT",
					strerror(errno));
				return FALSE;
			}
		}

		(void) memset(&req, 0, sizeof(req));
		(void) memset(&rep, 0, sizeof(rep));

		req.header.pack_len = sizeof(struct sg_header);
		req.header.reply_len = datalen + sizeof(struct sg_header);
		req.header.result = 0;
		req.header.twelve_byte = (int) (cmdlen == 12);
	
		(void) memcpy(&req.bytes[0], cmdpt, cmdlen);
		if (datalen > 0) {
			if (datalen > SG_BIG_BUFF) {
				DBGPRN(DBG_DEVIO)(errfp,
					"SCSI command failure: %s\n",
					"I/O data size too large");
				return FALSE;
			}

			if (datapt != NULL && rw == OP_DATAOUT) {
				(void) memcpy(&req.bytes[cmdlen],
					      datapt, datalen);
			}
		}

		/* Send command */
		if (rw == OP_DATAOUT)
			reqsz = sizeof(struct sg_header) + cmdlen + datalen;
		else
			reqsz = sizeof(struct sg_header) + cmdlen;

		oint = signal(SIGINT, SIG_IGN);
		oquit = signal(SIGQUIT, SIG_IGN);
		opipe = signal(SIGPIPE, SIG_IGN);
		oterm = signal(SIGTERM, SIG_IGN);

		if ((ret = write(fd, &req, reqsz)) < 0) {
			DBGPRN(DBG_DEVIO)(errfp,
				"SG: write failure: %s\n",
				strerror(errno));
		}
		else if (ret != reqsz) {
			DBGPRN(DBG_DEVIO)(errfp,
				"SG: Wrote %d bytes, expected to write %d.\n",
				ret, reqsz);
		}

		/* Get command response */
		if (read(fd, &rep, sizeof(sg_request_t)) < 0) {
			DBGPRN(DBG_DEVIO)(errfp,
				"SCSI command failure: SG: read failure: %s\n",
				strerror(errno));
			(void) signal(SIGINT, oint);
			(void) signal(SIGQUIT, oquit);
			(void) signal(SIGPIPE, opipe);
			(void) signal(SIGTERM, oterm);
			return FALSE;
		}

		(void) signal(SIGINT, oint);
		(void) signal(SIGQUIT, oquit);
		(void) signal(SIGPIPE, opipe);
		(void) signal(SIGTERM, oterm);

		if (rep.header.result != 0) {
			DBGPRN(DBG_DEVIO)(errfp, "SCSI command failure: "
				"%s=0x%02x %s=0x%x %s=0x%x %s=0x%x %s=0x%x\n",
				"Opcode", cmdpt[0],
				"Status", status_byte(rep.header.result),
				"Msg", msg_byte(rep.header.result),
				"Host", host_byte(rep.header.result),
				"Driver", driver_byte(rep.header.result));

			rp = (req_sense_data_t *)(void *)
					&rep.header.sense_buffer;
			if (rp->key != 0) {
				(void) fprintf(errfp,
				    "Key=0x%x Code=0x%x Qual=0x%x\n",
				    rp->key, rp->code, rp->qual);
			}

			return FALSE;
		}

		if (datapt != NULL && rw == OP_DATAIN && datalen > 0)
			(void) memcpy(datapt, rep.bytes, datalen);
	}

	return TRUE;
}


/*
 * pthru_open
 *	Open SCSI pass-through device
 *
 * Args:
 *	path - device path name string
 *
 * Return:
 *	Device file descriptor, or -1 on failure.
 */
int
pthru_open(char *path)
{
	int		i,
			fd,
			sg_idx,
			mode;
	word32_t	regflag;
	bool_t		is_sg;
	struct stat	stbuf;
	char		*usepath,
			tmppath[12],
			errstr[ERR_BUF_SZ],
			buf[SCSI_IOCTL_MAX_BUF];

	/* Check for validity of device node */
	if (stat(path, &stbuf) < 0) {
		(void) sprintf(errstr, app_data.str_staterr, path);
		DI_FATAL(errstr);
		return -1;
	}

	/* Linux CD-ROM device is a block special file, but the SCSI generic
	 * device (for medium changer support) is a character special file.
	 */
	if (S_ISCHR(stbuf.st_mode) &&
	    MAJOR(stbuf.st_rdev) == SCSI_GENERIC_MAJOR)
		is_sg = TRUE;
	else if (S_ISBLK(stbuf.st_mode))
		is_sg = FALSE;
	else {
		(void) sprintf(errstr, app_data.str_noderr, path);
		DI_FATAL(errstr);
		return -1;
	}

	if (is_sg) {
		/* The device path is a SCSI generic device */
		usepath = path;
		regflag |= 0x1;
	}
	else {
		sg_idx = find_sg(path, (int) MINOR(stbuf.st_rdev));
		if (sg_idx >= 0) {
			/* Check /dev/sg{n} where n is a number */
			regflag |= 0x1;
			sprintf(tmppath, "/dev/sg%d", sg_idx);
			if (stat(tmppath, &stbuf) < 0) {
				/* Check /dev/sg{n} where n is a letter */
				sprintf(tmppath, "/dev/sg%c", sg_idx + 'a');
				if (stat(tmppath, &stbuf) < 0)
					regflag &= ~0x1;
			}

			if ((regflag & 0x1) != 0 && S_ISCHR(stbuf.st_mode)) {
				usepath = tmppath;
			}
			else {
				usepath = path;
				regflag &= ~0x1;
			}
		}
		else {
			usepath = path;
			regflag &= ~0x1;
		}
	}

	if ((regflag & 0x1) == 0) {
		/* Use SCSI ioctl method */
		DBGPRN(DBG_DEVIO)(errfp,
			"Using SCSI ioctl interface on %s\n",
			path);
		mode = O_RDONLY | O_EXCL | O_NONBLOCK;
	}
	else {
		/* Use SCSI generic method */
		DBGPRN(DBG_DEVIO)(errfp,
			"Using SCSI Generic interface for %s: %s\n",
			path, usepath);
		mode = O_RDWR | O_EXCL | O_NONBLOCK;
	}

	sync();

	if ((fd = open(usepath, mode)) < 0) {
		DBGPRN(DBG_DEVIO)(errfp,
			"Cannot open %s: errno=%d\n", usepath, errno);
		return -1;
	}

	if ((regflag & 0x1) != 0) {
		/* Eat any unwanted garbage */
		i = fcntl(fd, F_GETFL);
		(void) fcntl(fd, F_SETFL, i | O_NONBLOCK);
		while (read(fd, buf, sizeof(buf)) != -1 || errno != EAGAIN)
			;	/* Empty buffer */
		(void) fcntl(fd, F_SETFL, i & ~O_NONBLOCK);
	}

	(void) di_devreg(fd, usepath, regflag);

	/* Linux hack:  The CD-ROM driver allows the open to succeed
	 * even if there is no CD loaded.  We test for the existence of
	 * a disc with scsipt_disc_present().
	 */
	if (!scsipt_disc_present(fd)) {
		/* No CD loaded */
		pthru_close(fd);
		return -1;
	}

	return (fd);
}


/*
 * pthru_close
 *	Close SCSI pass-through device
 *
 * Args:
 *	fd - Device file descriptor
 *
 * Return:
 *	Nothing.
 */
void
pthru_close(int fd)
{
	int		i;
	di_devreg_t	*regp;
	byte_t		buf[SCSI_IOCTL_MAX_BUF];

	if ((regp = di_devgetent(fd)) == NULL)
		return;

	di_devunreg(fd);
	(void) close(fd);
}


/*
 * pthru_vers
 *	Return OS Interface Module version string
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Module version text string.
 */
char *
pthru_vers(void)
{
	return ("OS Interface module for Linux");
}

#endif	/* linux DI_SCSIPT DEMO_ONLY */

