
/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "cdaudio.h"
#include <pthread.h>
#include "xmms/i18n.h"
#include "libxmms/util.h"

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#elif defined HAVE_SYS_CDIO_H
#include <sys/cdio.h>
#endif

static char * cdda_get_title(cdda_disc_toc_t *toc, gint track);

static void cdda_init(void);
static int is_our_file(char *filename);
static GList *scan_dir(char *dir);
static void play_file(char *filename);
static void stop(void);
static void cdda_pause(short p);
static void seek(int time);
static int get_time(void);
static void get_song_info(char *filename, char **title, int *length);
static void get_volume(int *l, int *r);
static void set_volume(int l, int r);
static void cleanup(void);
void cdda_fileinfo(gchar *filename);

InputPlugin cdda_ip =
{
	NULL,
	NULL,
	NULL, /* Description */
	cdda_init,
	NULL,				/* about */
	cdda_configure,
	is_our_file,
	scan_dir,
	play_file,
	stop,
	cdda_pause,
	seek,
	NULL,				/* set_eq */
	get_time,
	get_volume,
	set_volume,
	cleanup,
	NULL,				/* obsolete */
	NULL,				/* add_vis_pcm */
	NULL,				/* set_info, filled in by xmms */
	NULL,				/* set_info_text, filled in by xmms */
	get_song_info,
	NULL, /*  cdda_fileinfo, */	/* file_info_box */
	NULL				/* output plugin handle */
};

CDDAConfig cdda_cfg;

static gint cdda_fd = -1;
static gint track;
static gboolean is_paused;
static cdda_disc_toc_t cd_toc;

/* Time to delay stop command in 1/10 second */
#define STOP_DELAY 20

static struct {
	gboolean playing;
	gboolean exit;
	gint threads;
} stop_info;

static pthread_mutex_t stop_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_t stop_thread;

#define STOP_LOCK() pthread_mutex_lock(&stop_mutex)
#define STOP_UNLOCK() pthread_mutex_unlock(&stop_mutex)


#if !defined(CDROMVOLREAD) && !(defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__))
static gint volume_left = 100, volume_right = 100;
#endif

InputPlugin *get_iplugin_info(void)
{
	cdda_ip.description = g_strdup_printf(_("CD Audio Player %s"), VERSION);
	return &cdda_ip;
}

static void cdda_init(void)
{
	ConfigFile *cfgfile;

	memset(&cdda_cfg, 0, sizeof(CDDAConfig));
	       
#if defined(HAVE_SYS_SOUNDCARD_H) || defined(HAVE_MACHINE_SOUNDCARD_H)
	cdda_cfg.use_oss_mixer = TRUE;
#endif

	if ((cfgfile = xmms_cfg_open_default_file()))
	{
		xmms_cfg_read_string(cfgfile, "CDDA", "device", &cdda_cfg.device);
		xmms_cfg_read_string(cfgfile, "CDDA", "directory", &cdda_cfg.directory);
		xmms_cfg_read_boolean(cfgfile, "CDDA", "use_oss_mixer", &cdda_cfg.use_oss_mixer);
		xmms_cfg_read_string(cfgfile, "CDDA", "name_format", &cdda_cfg.name_format);
		xmms_cfg_read_boolean(cfgfile, "CDDA", "use_cddb", &cdda_cfg.use_cddb);
		xmms_cfg_read_string(cfgfile, "CDDA", "cddb_server", &cdda_cfg.cddb_server);
		xmms_cfg_read_int(cfgfile, "CDDA", "cddb_protocol_level", &cdda_cfg.cddb_protocol_level);
#ifdef WITH_CDINDEX
		xmms_cfg_read_boolean(cfgfile, "CDDA", "use_cdin", &cdda_cfg.use_cdin);
#else
		cdda_cfg.use_cdin = FALSE;
#endif
		xmms_cfg_read_string(cfgfile, "CDDA", "cdin_server", &cdda_cfg.cdin_server);
		xmms_cfg_free(cfgfile);
	}
#ifdef HAVE_SYS_CDIO_H
# ifdef __FreeBSD__
	if (!cdda_cfg.device)
		cdda_cfg.device = g_strdup("/dev/acd0c");
	if (!cdda_cfg.directory)
		cdda_cfg.directory = g_strdup("/cdrom");
# else
	if (!cdda_cfg.device)
		cdda_cfg.device = g_strdup("/vol/dev/aliases/cdrom0");
	if (!cdda_cfg.directory)
		cdda_cfg.directory = g_strdup("/cdrom/cdrom0");
# endif
#else
	if (!cdda_cfg.device)
		cdda_cfg.device = g_strdup("/dev/cdrom");
	if (!cdda_cfg.directory)
		cdda_cfg.directory = g_strdup("/mnt/cdrom/");
#endif
	if (!cdda_cfg.cdin_server)
		cdda_cfg.cdin_server = g_strdup("www.cdindex.org");
	if (!cdda_cfg.cddb_server)
		cdda_cfg.cddb_server = g_strdup(CDDB_DEFAULT_SERVER);
	if (!cdda_cfg.name_format)
		cdda_cfg.name_format = g_strdup("%p - %t");
}

static void cleanup(void)
{
	STOP_LOCK();
	while (stop_info.threads > 0)
	{
		STOP_UNLOCK();
		xmms_usleep(10000);
		STOP_LOCK();
	}
	STOP_UNLOCK();
}

static int is_our_file(char *filename)
{
	gchar *ext;

	ext = strrchr(filename, '.');
	if (ext && !strcasecmp(ext, ".cda"))
		return TRUE;

	return FALSE;
}

static gboolean is_mounted(gchar * device_name)
{
#ifdef HAVE_MNTENT_H
	FILE *mounts;
	struct mntent *mnt;
	char devname[256];
	struct stat st;

	if (lstat(device_name, &st) < 0)
		return -1;

	if (S_ISLNK(st.st_mode))
		readlink(device_name, devname, 256);
	else
		strncpy(devname, device_name, 256);

	if ((mounts = setmntent(MOUNTED, "r")) == NULL)
		return TRUE;

	while ((mnt = getmntent(mounts)) != NULL)
	{
		if (strcmp(mnt->mnt_fsname, devname) == 0)
		{
			endmntent(mounts);
			return TRUE;
		}
	}
	endmntent(mounts);
#endif
	return FALSE;
}

static GList *scan_dir(char *dir)
{
	GList *list = NULL;
	gint i;
	cdda_disc_toc_t toc;

	if (strncmp(cdda_cfg.directory, dir, strlen(cdda_cfg.directory)))
		return NULL;

	if (!cdda_get_toc(&toc))
		return NULL;

	for (i = toc.last_track; i >= toc.first_track; i--)
		if (!toc.track[i].flags.data_track)
			list = g_list_prepend(list, g_strdup_printf("Track %02d.cda", i));

	return list;
}

guint cdda_calculate_track_length(cdda_disc_toc_t *toc, gint track)
{
	if (track == toc->last_track)
		return (LBA(toc->leadout) - LBA(toc->track[track]));
	else
		return (LBA(toc->track[track + 1]) - LBA(toc->track[track]));
}

static void play_file(char *filename)
{
	gchar *tmp;

	if (is_mounted(cdda_cfg.device))
		return;

	tmp = strrchr(filename, '/');
	if (tmp)
		tmp++;
	else
		tmp = filename;

	if (!sscanf(tmp, "Track %d.cda", &track))
		return;

	if ((cdda_fd = open(cdda_cfg.device, CDOPENFLAGS)) == -1)
		return;

	if (!cdda_get_toc(&cd_toc) || cd_toc.track[track].flags.data_track ||
	    track < cd_toc.first_track || track > cd_toc.last_track)
	{
		close(cdda_fd);
		cdda_fd = -1;
		return;
	}
	cdda_ip.set_info(cdda_get_title(&cd_toc, track),
			 (cdda_calculate_track_length(&cd_toc, track) * 1000) / 75,
			 44100 * 2 * 2 * 8, 44100, 2);

	is_paused = FALSE;
	STOP_LOCK();
	stop_info.playing = TRUE;
	STOP_UNLOCK();

	seek(0);
}

static char * cdda_get_title(cdda_disc_toc_t *toc, gint track)
{
	static guint32 cached_id;
	static cdinfo_t cdinfo;
	static pthread_mutex_t title_mutex = PTHREAD_MUTEX_INITIALIZER;
	guint32 disc_id;
	gchar *title;

	disc_id = cdda_cddb_compute_discid(toc);
	
	/*
	 * We want to avoid looking up a album from two threads simultaneously.
	 * This can happen since we are called both from the main-thread and
	 * from the playlist-thread.
	 */
	
	pthread_mutex_lock(&title_mutex);
	if (!(disc_id == cached_id && cdinfo.is_valid))
	{
		cdda_cdinfo_flush(&cdinfo);
		cached_id = disc_id;

		if (!cdda_cdinfo_read_file(disc_id, &cdinfo))
		{
			if (cdda_cfg.use_cdin)
				cdda_cdindex_get_idx(toc, &cdinfo);
			if (cdda_cfg.use_cddb && !cdinfo.is_valid)
				cdda_cddb_get_info(toc, &cdinfo);
			if (cdinfo.is_valid)
				cdda_cdinfo_write_file(disc_id, &cdinfo);
		}
	}
	pthread_mutex_unlock(&title_mutex);
	
	title = cdda_cdinfo_get_name(&cdinfo, track, cdda_cfg.name_format);
	if (!title)
		title = g_strdup_printf(_("CD Audio Track %02u"), track);
	return title;
}

static void * stop_func(void *arg)
{
	/*
	 * This is run as a separate thread,
	 * and stop the cdrom after the stop delay
	 */
	gint i;
	gint fd;

	STOP_LOCK();

	for (i = 0; i < STOP_DELAY; i++)
	{
		STOP_UNLOCK();
		xmms_usleep(100000);
		STOP_LOCK();
		if (stop_info.playing)
		{
			stop_info.threads--;
			STOP_UNLOCK();
			return NULL;
		}
		if (stop_info.exit)
			break;
	}

	fd = open(cdda_cfg.device, CDOPENFLAGS);
	if (fd != -1)
	{
#ifdef CDROMSTOP
		ioctl(fd, CDROMSTOP, 0);
#endif
#ifdef CDIOCSTOP
		ioctl(fd, CDIOCSTOP, 0);
#endif
		close(fd);
	}
	stop_info.threads--;
	STOP_UNLOCK();
	return NULL;
}

static void stop(void)
{
	static pthread_attr_t *attr = NULL;
	
	if (cdda_fd < 0)
		return;

	if (!attr)
	{
		attr = g_malloc(sizeof(pthread_attr_t));
		pthread_attr_init(attr);
		pthread_attr_setdetachstate(attr, PTHREAD_CREATE_DETACHED);
	}
	
#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
/*  	ioctl(cdda_fd, CDIOCSTOP, 0); */
	ioctl(cdda_fd, CDIOCPAUSE, 0);
#else
/*  	ioctl(cdda_fd, CDROMSTOP, 0); */
	ioctl(cdda_fd, CDROMPAUSE);
#endif
	close(cdda_fd);
	cdda_fd = -1;
	STOP_LOCK();
	stop_info.threads++;
	stop_info.playing = FALSE;
	STOP_UNLOCK();
	pthread_create(&stop_thread, attr, stop_func, NULL);
}

static void cdda_pause(short p)
{
#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
	ioctl(cdda_fd, p ? CDIOCPAUSE : CDIOCRESUME);
#else
	ioctl(cdda_fd, p ? CDROMPAUSE : CDROMRESUME);
#endif
	is_paused = (p ? TRUE : FALSE);
}

static void seek(int time)
{
	struct cdda_msf *end, start;

#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
	struct ioc_play_msf msf;
#else
	struct cdrom_msf msf;
#endif

	start.minute = (cd_toc.track[track].minute * 60 +
			cd_toc.track[track].second + time) / 60;
	start.second = (cd_toc.track[track].second + time) % 60;
	start.frame = cd_toc.track[track].frame;
	if (track == cd_toc.last_track)
		end = &cd_toc.leadout;
	else
		end = &cd_toc.track[track + 1];

#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
	msf.start_m = start.minute;
	msf.start_s = start.second;
	msf.start_f = start.frame;
	msf.end_m = end->minute;
	msf.end_s = end->second;
	msf.end_f = end->frame;
	ioctl(cdda_fd, CDIOCPLAYMSF, &msf);
#else
	msf.cdmsf_min0 = start.minute;
	msf.cdmsf_sec0 = start.second;
	msf.cdmsf_frame0 = start.frame;
	msf.cdmsf_min1 = end->minute;
	msf.cdmsf_sec1 = end->second;
	msf.cdmsf_frame1 = end->frame;
	ioctl(cdda_fd, CDROMPLAYMSF, &msf);
#endif
	if (is_paused)
		cdda_pause(TRUE);
}

static int get_time(void)
{
	gint frame, start_frame, length;
#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
	struct ioc_read_subchannel subchnl;
	struct cd_sub_channel_info subinfo;

	if (cdda_fd == -1)
		return -1;

	subchnl.address_format = CD_MSF_FORMAT;
	subchnl.data_format = CD_CURRENT_POSITION;
	subchnl.data_len = sizeof(subinfo);
	subchnl.data = &subinfo;
	ioctl(cdda_fd, CDIOCREADSUBCHANNEL, &subchnl);

	frame = LBA(subchnl.data->what.position.absaddr.msf);
#else
	struct cdrom_subchnl subchnl;

	if (cdda_fd == -1)
		return -1;

	subchnl.cdsc_format = CDROM_MSF;
	ioctl(cdda_fd, CDROMSUBCHNL, &subchnl);

	frame = LBA(subchnl.cdsc_absaddr.msf);
#endif
	start_frame = LBA(cd_toc.track[track]);
	length = cdda_calculate_track_length(&cd_toc, track);

	if (frame - start_frame >= length - 20)		/* 20 seems to work better */
		return -1;

	return ((frame - start_frame) * 1000) / 75;
}

static void get_song_info(char *filename, char **title, int *len)
{
	cdda_disc_toc_t toc;
	gint t;
	gchar *tmp;

	*title = NULL;
	*len = -1;

	tmp = strrchr(filename, '/');
	if (tmp)
		tmp++;
	else
		tmp = filename;
	
	if (!sscanf(tmp, "Track %d.cda", &t))
		return;
	if (!cdda_get_toc(&toc))
		return;
	if (t < toc.first_track || t > toc.last_track || toc.track[t].flags.data_track)
		return;
	
	*len = (cdda_calculate_track_length(&toc, t) * 1000) / 75;
	*title = cdda_get_title(&toc, t);
}

static void get_volume(int *l, int *r)
{
	int fd, devs, cmd, v;

#if defined(HAVE_SYS_SOUNDCARD_H) || defined(HAVE_MACHINE_SOUNDCARD_H)
	if (cdda_cfg.use_oss_mixer)
	{
		fd = open(DEV_MIXER, O_RDONLY);
		if (fd != -1)
		{
			ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devs);
			if (devs & SOUND_MASK_CD)
				cmd = SOUND_MIXER_READ_CD;
			else if (devs & SOUND_MASK_VOLUME)
				cmd = SOUND_MIXER_READ_VOLUME;
			else
			{
				close(fd);
				return;
			}
			ioctl(fd, cmd, &v);
			*r = (v & 0xFF00) >> 8;
			*l = (v & 0x00FF);
			close(fd);
		}
	}
	else
#endif
	if (!cdda_cfg.use_oss_mixer)
	{

#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
		struct ioc_vol vol;

		if (cdda_fd != -1)
		{
			ioctl(cdda_fd, CDIOCGETVOL, &vol);
			*l = (100 * vol.vol[0]) / 255;
			*r = (100 * vol.vol[1]) / 255;
		}
#elif defined(CDROMVOLREAD)
		struct cdrom_volctrl vol;

		if (cdda_fd != -1)
		{
			ioctl(cdda_fd, CDROMVOLREAD, &vol);
			*l = (100 * vol.channel0) / 255;
			*r = (100 * vol.channel1) / 255;
		}
#else
		*l = volume_left;
		*r = volume_right;
#endif

	}
}

static void set_volume(int l, int r)
{
#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
	struct ioc_vol vol;
#else
	struct cdrom_volctrl vol;
#endif
	int fd, devs, cmd, v;

#if defined(HAVE_SYS_SOUNDCARD_H) || defined(HAVE_MACHINE_SOUNDCARD_H)
	if (cdda_cfg.use_oss_mixer)
	{
		fd = open(DEV_MIXER, O_RDONLY);
		if (fd != -1)
		{
			ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devs);
			if (devs & SOUND_MASK_CD)
				cmd = SOUND_MIXER_WRITE_CD;
			else if (devs & SOUND_MASK_VOLUME)
				cmd = SOUND_MIXER_WRITE_VOLUME;
			else
			{
				close(fd);
				return;
			}
			v = (r << 8) | l;
			ioctl(fd, cmd, &v);
			close(fd);
		}
	}
	else
#endif
	{
		if (cdda_fd != -1)
		{
#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
			vol.vol[0] = vol.vol[2] = (l * 255) / 100;
			vol.vol[1] = vol.vol[3] = (r * 255) / 100;
			ioctl(cdda_fd, CDIOCSETVOL, &vol);
#else
			vol.channel0 = vol.channel2 = (l * 255) / 100;
			vol.channel1 = vol.channel3 = (r * 255) / 100;
			ioctl(cdda_fd, CDROMVOLCTRL, &vol);
#endif
		}
#if !defined(CDROMVOLREAD) && !(defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__))
		volume_left = l;
		volume_right = r;
#endif
	}
}

gboolean cdda_get_toc(cdda_disc_toc_t *info)
{
	gboolean retv = FALSE;

#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
	struct ioc_toc_header tochdr;
	struct ioc_read_toc_single_entry tocentry;
#else	
	struct cdrom_tochdr tochdr;
	struct cdrom_tocentry tocentry;
#endif
	gint i;
	gint fd;

	if (is_mounted(cdda_cfg.device))
		return FALSE;

	if ((fd = open(cdda_cfg.device, CDOPENFLAGS)) == -1)
		return FALSE;

	memset(info, 0, sizeof(cdda_disc_toc_t));

#if defined(HAVE_SYS_CDIO_H) && defined(__FreeBSD__)
	if ( ioctl(fd, CDIOREADTOCHEADER, &tochdr) )
	    goto done;

	for (i = tochdr.starting_track; i <= tochdr.ending_track; i++)
	{		
		tocentry.address_format = CD_MSF_FORMAT;
		tocentry.track = i;
		if (ioctl(fd, CDIOREADTOCENTRY, &tocentry))
			goto done;
		info->track[i].minute =
		    tocentry.entry.addr.msf.minute;
		info->track[i].second =
		    tocentry.entry.addr.msf.second;
		info->track[i].frame =
		    tocentry.entry.addr.msf.frame;
		info->track[i].flags.data_track =
		    tocentry.entry.control & 4 == 4;
	}

	/* Get the leadout track */
	tocentry.track = 0xAA;
	tocentry.address_format = CD_MSF_FORMAT;
	if (ioctl(fd, CDIOREADTOCENTRY, &tocentry))
		goto done;
	info->leadout.minute = tocentry.entry.addr.msf.minute;
	info->leadout.second = tocentry.entry.addr.msf.second;
	info->leadout.frame = tocentry.entry.addr.msf.frame;
	
	info->first_track = tochdr.starting_track;
	info->last_track = tochdr.ending_track;
	retv = TRUE;

#else
	if (ioctl(fd, CDROMREADTOCHDR, &tochdr))
		goto done;

	for (i = tochdr.cdth_trk0; i <= tochdr.cdth_trk1; i++)
	{		
		tocentry.cdte_format = CDROM_MSF;
		tocentry.cdte_track = i;
		if (ioctl(fd, CDROMREADTOCENTRY, &tocentry))
			goto done;
		info->track[i].minute = tocentry.cdte_addr.msf.minute;
		info->track[i].second = tocentry.cdte_addr.msf.second;
		info->track[i].frame = tocentry.cdte_addr.msf.frame;
		info->track[i].flags.data_track = tocentry.cdte_ctrl == CDROM_DATA_TRACK;
	}

	/* Get the leadout track */
	tocentry.cdte_track = CDROM_LEADOUT;
	tocentry.cdte_format = CDROM_MSF;
	if (ioctl(fd, CDROMREADTOCENTRY, &tocentry))
		goto done;
	info->leadout.minute = tocentry.cdte_addr.msf.minute;
	info->leadout.second = tocentry.cdte_addr.msf.second;
	info->leadout.frame = tocentry.cdte_addr.msf.frame;
	
	info->first_track = tochdr.cdth_trk0;
	info->last_track = tochdr.cdth_trk1;
	retv = TRUE;
#endif

 done:
	close(fd);

	return retv;
}
