/*
 *  Copyright 2000 Hvard Kvlen <havardk@sol.no>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "xmms/plugin.h"
#include "xmms/i18n.h"
#include "libxmms/util.h"
#include "config.h"
#include <glib.h>
#include <pthread.h>
#include <string.h>
#include <math.h>
#include <stdio.h>

#define MIN_FREQ 10
#define MAX_FREQ 20000

static InputPlugin tone_ip;

static gboolean going;
static gboolean audio_error;
static pthread_t play_thread;

static void tone_about(void)
{
	static GtkWidget *box;
	box = xmms_show_message(
		_("About Tone Generator"),
		_("Sinus tone generator by Hvard Kvlen <havardk@xmms.org>\n\n"
		  "To use it, add a URL: tone://frequency\n"
		  "e.g. tone://2000 to play a 2000Hz tone"), _("Ok"),
		FALSE, NULL, NULL);
	gtk_signal_connect(GTK_OBJECT(box), "destroy",
			   gtk_widget_destroyed, &box);
}

static int tone_is_our_file(char *filename)
{
	if (!strncmp(filename, "tone://", 7))
		return TRUE;
	return FALSE;
}

#define BUF_SAMPLES 512
#define BUF_BYTES BUF_SAMPLES * 2

static void* play_loop(void *arg)
{
	gint16 data[BUF_SAMPLES];
	gint freq;
	gdouble wd;
	gint i, t = 0;
	

	freq = GPOINTER_TO_INT(arg);
	wd = ((2 * 3.14159265359 * freq) / 44100);

	while (going)
	{
		for (i = 0; i < BUF_SAMPLES; i++)
			data[i] = GINT16_TO_LE(((1 << 15) - 1) * sin(wd * t++));
		if (t > 44100)
			t -= 44100;
		tone_ip.add_vis_pcm(tone_ip.output->written_time(),
				    FMT_S16_LE, 1, BUF_BYTES, data);
		while(tone_ip.output->buffer_free() < BUF_BYTES && going)
			xmms_usleep(30000);
		if (going)
			tone_ip.output->write_audio(data, BUF_BYTES);
	}
	/* Make sure the output plugin stops prebuffering */
	tone_ip.output->buffer_free();
	tone_ip.output->buffer_free();
	pthread_exit(NULL);
}

static void tone_play(char *filename)
{
	gint freq;
	gchar *name;
	
	if (!(sscanf(filename, "tone://%d", &freq) == 1 &&
	      freq >= MIN_FREQ && freq <= MAX_FREQ))
		return;
	
	going = TRUE;
	audio_error = FALSE;
	if (tone_ip.output->open_audio(FMT_S16_LE, 44100, 1) == 0)
	{
		audio_error = TRUE;
		going = FALSE;
		return;
	}

	name = g_strdup_printf(_("Tone generator: %d Hz"), freq);
	tone_ip.set_info(name, -1, 16 * 44100, 44100, 1);
	g_free(name);
	pthread_create(&play_thread, NULL, play_loop, GINT_TO_POINTER(freq));
}

static void tone_stop(void)
{
	if (going)
	{
		going = FALSE;
		pthread_join(play_thread, NULL);
		tone_ip.output->close_audio();
	}
}

static void tone_pause(short paused)
{
	tone_ip.output->pause(paused);
}

static int tone_get_time(void)
{
	if (audio_error)
		return -2;
	if (!going && !tone_ip.output->buffer_playing())
		return -1;
	return tone_ip.output->output_time();
}

static void tone_song_info(char *filename, char **title, int *length)
{
	int freq;
	*length = -1;
	*title = NULL;
	if (sscanf(filename, "tone://%d", &freq) == 1 &&
	    freq >= MIN_FREQ && freq <= MAX_FREQ)
		*title = g_strdup_printf(_("Tone generator: %d Hz"), freq);
}
		
	
	
static InputPlugin tone_ip = 
{
	NULL,
	NULL,
	NULL, /* Description */
	NULL,
	tone_about,
	NULL,
	tone_is_our_file,
	NULL,
	tone_play,
	tone_stop,
	tone_pause,
	NULL,
	NULL,
	tone_get_time,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
	tone_song_info,
	NULL,
	NULL
};

InputPlugin *get_iplugin_info(void)
{
	tone_ip.description =
		g_strdup_printf(_("Tone Generator %s"), VERSION);
	return &tone_ip;
}
