/*
Vorbis xmms module, tested with xmms 0.9.5.1 and xmms CVS source.
Report problems to Tony Arcieri, <bascule@inferno.tusculum.edu>
*/

/* Note that this uses vorbisfile.a, which is not (currently)
thread-safe.  Right now, we only make vorbisfile calls from one thread
so no problem.  Looks like I better at least add locking to
vorbisfile.a ASAP (it won't require any API changes) --Monty 19991112 */

#define _REENTRANT 1
#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <ogg/ogg.h>
#include <vorbis/codec.h>
#include <vorbis/vorbisfile.h>

#include "xmms/util.h"
#include "xmms/plugin.h"
#include "libxmms/util.h"


#define BLOCK_SIZE 4096

static int vorbis_check_file(char *filename);
static void vorbis_play(char *filename);
static void vorbis_stop(void);
static void vorbis_pause(short p);
static void vorbis_seek(int time);
static int vorbis_time(void);
static void vorbis_get_song_info(char *filename, char **title, int *length);
static gchar *vorbis_generate_title(vorbis_comment *comment, char *fn);

InputPlugin vorbis_ip =
{
   NULL,
   NULL,
   "OggVorbis Player 0.1",
   NULL,
   NULL,
   NULL,
   vorbis_check_file,
   NULL,
   vorbis_play,
   vorbis_stop,
   vorbis_pause,
   vorbis_seek,
   NULL,
   vorbis_time,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL,
   NULL,
   vorbis_get_song_info,
   NULL,
   NULL
};

static OggVorbis_File vf;
static pthread_t tid;
static int vorbis_playing = 0, vorbis_eos = 0;
static long timeoffset=0;
static long seekneeded=-1;
static pthread_mutex_t vf_mutex = PTHREAD_MUTEX_INITIALIZER;
	

InputPlugin 
*get_iplugin_info(void)
{
   return &vorbis_ip;
}

static int 
vorbis_check_file(char *filename)
{
   FILE *stream;
   OggVorbis_File vf; /* avoid thread interaction */

   /* This isn't supported... yet */
   if(strncasecmp(filename, "http://", 7) == 0)
     return FALSE;
   
   if((stream = fopen(filename, "r")) == NULL)
     return FALSE;
   
   /* The open function performs full stream detection and machine
      initialization.  If it returns zero, the stream *is* Vorbis and
      we're fully ready to decode. */
   if(ov_open(stream, &vf, NULL, 0) < 0) {
      fclose(stream);
      return FALSE;
   }

   ov_clear(&vf); /* once the ov_open succeeds, the stream belongs to
                     vorbisfile.a.  ov_clear will fclose it */
   return TRUE;
}

static void
*vorbis_play_loop(void *arg)
{
   char *filename = (char *)arg;
   char pcmout[4096];
   vorbis_comment *comment;
   gchar *title=NULL;
   double time;
   long timercount=0;

   int last_section=-1;

   FILE *stream=NULL;
   vorbis_info *vi=NULL;

   pthread_mutex_lock(&vf_mutex);
   memset(&vf,0,sizeof(vf));
   if((stream = fopen(filename, "r")) == NULL)
   {
     pthread_mutex_unlock(&vf_mutex);
     vorbis_eos = 1;
     goto play_cleanup;
   }

   /* The open function performs full stream detection and machine
      initialization.  None of the rest of ov_xx() works without it */

   if(ov_open(stream, &vf, NULL, 0) < 0){     
     fclose(stream);
     pthread_mutex_unlock(&vf_mutex);
     vorbis_eos = 1;
     goto play_cleanup;
   }
   vi=ov_info(&vf,-1);
   time=ov_time_total(&vf, -1) * 1000;

   if(vi->channels > 2)
   {
	   goto play_cleanup;
   }
	   
   
   vorbis_ip.set_info(filename, time, ov_bitrate(&vf, -1), vi->rate, vi->channels);
   pthread_mutex_unlock(&vf_mutex);

   while(strchr(filename, '/') != NULL)
     filename = strchr(filename, '/') + 1;   

   seekneeded = -1;
   
   /* Note that chaining changes things here; A vorbis file may be a
      mix of different channels, bitrates and sample rates.  You can
      fetch the information for any section of the file using the ov_
      interface.  */

   while(vorbis_playing) {
     int ret;
     int current_section=-1; /* A vorbis physical bitstream may
                                consist of many logical sections
                                (information for each of which may be
                                fetched from the vf structure).  This
                                value is filled in by ov_read to alert
                                us what section we're currently
                                decoding in case we need to change
                                playback settings at a section
                                boundary */
	 if(seekneeded != -1)
	 {
     	pthread_mutex_lock(&vf_mutex);

		/* We need to guard against seeking to the end, or things don't work right.
		 * Instead, just seek to one second prior to this */
		if(seekneeded == (long)ov_time_total(&vf,-1))
			seekneeded--;


	 	timeoffset = seekneeded;
		timercount = 0; /* reset this so that bitrate gets updated as needed */
		seekneeded = -1;
	 	vorbis_ip.output->flush(timeoffset);
	 	ov_time_seek(&vf, timeoffset);

		vorbis_eos = FALSE;

	 	pthread_mutex_unlock(&vf_mutex);
	 }

     if(!vorbis_eos)
     {

	     pthread_mutex_lock(&vf_mutex);
	     ret=ov_read(&vf,pcmout,sizeof(pcmout),0,2,1,&current_section);
	     switch(ret){
	     case 0:
		     /* EOF */
		     pthread_mutex_unlock(&vf_mutex);

			 if(!vorbis_ip.output->buffer_playing())
		     	vorbis_eos = 1;
			 else
			    xmms_usleep(10000);

		     break;
	     case -1:
		     /* error in the stream.  Not a problem, just reporting it in
			case we (the app) cares.  In this case, we don't. */
		     pthread_mutex_unlock(&vf_mutex);			 
		     break;
	     default:
		     if(current_section!=last_section){
			     vi=ov_info(&vf,-1); /* The info struct is different in each
						    section.  vf holds them all for the
						    given bitstream.  This requests the
						    current one */

			     /* we don't do the audio open first just in case we seek()ed
				before beginning decode.  It's also more elegant to put
				the update and init in one place */

			     /* This resets the seek slider!  Thus we save the timeoffset
				below to add back to the slider when we hit the next link
				of a chain.*/
			     vorbis_ip.output->close_audio();
			     vorbis_ip.output->open_audio(FMT_S16_LE, vi->rate, vi->channels);
			     timeoffset=ov_time_tell(&vf);

			     /* set total play time, bitrate, rate, and channels of current
				section */
			     if(title)g_free(title);
			     comment = ov_comment(&vf, -1);
			     if(comment)
				     title = vorbis_generate_title(comment, filename);
			     if(!title)
				     title = g_strdup(filename);
	 
			     time= ov_time_total(&vf, -1) * 1000;
			     vorbis_ip.set_info(title,time, ov_bitrate(&vf, current_section), 
						vi->rate, vi->channels);
			     timercount=0;
			     
		     }else{
		       if(!(vi->bitrate_upper==vi->bitrate_lower==vi->bitrate_nominal)){
			 if(vorbis_ip.output->output_time()>timercount+500){ /* simple hack to avoid updating 
										 too often */
			   long br=ov_bitrate_instant(&vf);
			   if(br>-1)
			     vorbis_ip.set_info(title,time, br,
						vi->rate, vi->channels);
			   timercount=vorbis_ip.output->output_time();
			 }
		       }
		     }

		     pthread_mutex_unlock(&vf_mutex);
		     last_section=current_section;
       
		     vorbis_ip.add_vis_pcm(timeoffset*1000+vorbis_ip.output->written_time(), 
					   FMT_S16_LE, 
					   vi->channels, 
					   ret,
					   pcmout);
     
		     /* XXX This same spinlock already exists in the write_audio call... */
		     while(vorbis_ip.output->buffer_free() < ret && vorbis_playing)
			     xmms_usleep(10000);
		     vorbis_ip.output->write_audio(pcmout,ret);
		     break;
	     }
     }
     else
     {
	     xmms_usleep(20000);
     }
   }
   vorbis_ip.output->close_audio();
   /* fall through intentional */

 play_cleanup:
   pthread_mutex_lock(&vf_mutex);
   if(title)g_free(title);
   ov_clear(&vf); /* ov_clear closes the stream if its open.  Safe to
                     call on an uninitialized structure as long as
                     we've zeroed it */
   pthread_mutex_unlock(&vf_mutex);
   vorbis_playing = 0;
   /* Umm... even simple assignments like this need to be locked when
      threading.  Why? *because the assignment is not necessarily a
      single machine instruction!* I'm not going to fix this right now
      (and I suspect that this mistake is made in the XMMS source too
      )-: */
   
   pthread_exit(NULL);
}

static void
vorbis_play(char *filename)
{
   /* This isn't supported... yet */
   if(strncasecmp(filename, "http://", 7) == 0)
     return;
   
   vorbis_playing = 1;
   vorbis_eos = 0;
   pthread_create(&tid, NULL, vorbis_play_loop, (void *)filename);
}
 
static void
vorbis_stop(void)
{
   vorbis_playing = 0;
   pthread_join(tid, NULL);
   
   vorbis_ip.output->close_audio();
}

static void
vorbis_pause(short p)
{
  vorbis_ip.output->pause(p);
}

static int
vorbis_time(void)
{
  if(vorbis_eos && !vorbis_ip.output->buffer_playing())
	  return -1;
  return timeoffset * 1000 + vorbis_ip.output->output_time();
}

static void
vorbis_seek(int time)
{
	seekneeded = time;

}

static void
vorbis_get_song_info(char *filename, char **title, int *length)
{
   FILE *stream;
   OggVorbis_File vf; /* avoid thread interaction */
   vorbis_comment *comment;

   if((stream = fopen(filename, "r")) == NULL)
     return;
   
   /* The open function performs full stream detection and machine
      initialization.  If it returns zero, the stream *is* Vorbis and
      we're fully ready to decode. */
   if(ov_open(stream, &vf, NULL, 0) < 0) {
      fclose(stream);
      return;
   }

   /* Retrieve the length */
   *length = ov_time_total(&vf, -1) * 1000;

   *title = NULL;
   comment = ov_comment(&vf, -1);
   if(comment)
	   *title = vorbis_generate_title(comment, filename);
   
   ov_clear(&vf); /* once the ov_open succeeds, the stream belongs to
                     vorbisfile.a.  ov_clear will fclose it */
}

static gchar *vorbis_generate_title(vorbis_comment *comment, char *fn)
{
	gchar *displaytitle = NULL;
	gchar *artist, *title;

	title = vorbis_comment_query(comment, "title", 0);
	artist = vorbis_comment_query(comment, "artist", 0);

	if(artist && title)
		displaytitle = g_strdup_printf("%s - %s", artist, title);
	else if(artist)
		displaytitle = g_strdup_printf("%s - unknown title", artist);
	else if(title)
		displaytitle = g_strdup(title);
	else
		displaytitle = g_strdup_printf("%s (no title)", fn);

	return displaytitle;
}


