/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "xmms.h"
#include "libxmms/dirbrowser.h"

GtkWidget *playlistwin, *playlistwin_save_filesel = NULL;
GtkWidget *playlistwin_load_filesel = NULL, *playlistwin_url_window = NULL,
         *playlistwin_dir_browser;
GtkItemFactory *playlistwin_sort_menu, *playlistwin_sub_menu;

GdkPixmap *playlistwin_bg;
GdkBitmap *playlistwin_mask = NULL;
GdkGC *playlistwin_gc;
gboolean playlistwin_focus = FALSE, playlistwin_resizing = FALSE;
gint playlistwin_resize_x, playlistwin_resize_y, playlistwin_move_x, playlistwin_move_y;

PlayList_List *playlistwin_list = NULL;
PlaylistSlider *playlistwin_slider = NULL;
PButton *playlistwin_shade, *playlistwin_close;
TextBox *playlistwin_time_min, *playlistwin_time_sec, *playlistwin_info = NULL,
       *playlistwin_sinfo = NULL;
SButton *playlistwin_srew, *playlistwin_splay, *playlistwin_spause, *playlistwin_sstop;
SButton *playlistwin_sfwd, *playlistwin_seject;
SButton *playlistwin_sscroll_up, *playlistwin_sscroll_down;
extern TButton *mainwin_pl;

GList *playlistwin_wlist = NULL;

Vis *playlistwin_vis;

static gboolean playlistwin_vis_enabled = FALSE;

enum
{
	ADD_URL, ADD_DIR, ADD_FILE
};
enum
{
	SUB_MISC, SUB_ALL, SUB_CROP, SUB_SELECTED
};
enum
{
	SEL_INV, SEL_ZERO, SEL_ALL
};
enum
{
	MISC_SORT, MISC_FILEINFO, MISC_MISCOPTS
};
enum
{
	PLIST_NEW, PLIST_SAVE, PLIST_LOAD
};

void playlistwin_sort_menu_callback(gpointer cb_data, guint action, GtkWidget * w);
void playlistwin_sub_menu_callback(gpointer cb_data, guint action, GtkWidget * w);

enum
{
	PLAYLISTWIN_SORT_BYTITLE, PLAYLISTWIN_SORT_BYFILENAME, PLAYLISTWIN_SORT_BYPATH,
	PLAYLISTWIN_SORT_SEL_BYTITLE, PLAYLISTWIN_SORT_SEL_BYFILENAME,
	PLAYLISTWIN_SORT_SEL_BYPATH, PLAYLISTWIN_SORT_RANDOMIZE, PLAYLISTWIN_SORT_REVERSE
};

#define PLAYLISTWIN_SORT_MENU_ENTRIES 11

GtkItemFactoryEntry playlistwin_sort_menu_entries[] =
{
	{N_("/Sort List"), NULL, NULL, 0, "<Branch>"},
	{N_("/Sort List/By Title"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_BYTITLE, "<Item>"},
	{N_("/Sort List/By Filename"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_BYFILENAME, "<Item>"},
	{N_("/Sort List/By Path + Filename"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_BYPATH, "<Item>"},
	{N_("/Sort Selection"), NULL, NULL, 0, "<Branch>"},
	{N_("/Sort Selection/By Title"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_SEL_BYTITLE, "<Item>"},
	{N_("/Sort Selection/By Filename"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_SEL_BYFILENAME, "<Item>"},
	{N_("/Sort Selection/By Path + Filename"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_SEL_BYPATH, "<Item>"},
	{N_("/-"), NULL, NULL, 0, "<Separator>"},
	{N_("/Randomize List"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_RANDOMIZE, "<Item>"},
	{N_("/Reverse List"), NULL, playlistwin_sort_menu_callback, PLAYLISTWIN_SORT_REVERSE, "<Item>"},
};

enum
{
	PLAYLISTWIN_REMOVE_DEAD_FILES
};

#define PLAYLISTWIN_SUB_MENU_ENTRIES 1

GtkItemFactoryEntry playlistwin_sub_menu_entries[] =
{
	{N_("/Remove Dead Files"), NULL, playlistwin_sub_menu_callback, PLAYLISTWIN_REMOVE_DEAD_FILES, "<Item>"},
};

void playlistwin_draw_frame(void);

static void playlistwin_update_info(void)
{
	GList *list;
	PlaylistEntry *entry;
	gchar *text, *sel_text, *tot_text;
	gint selection = 0, total = 0;
	gboolean selection_more = FALSE, total_more = FALSE;

	PL_LOCK();
	list = get_playlist();
	while (list)
	{
		entry = list->data;
		if (entry->length != -1)
			total += entry->length;
		else
			total_more = TRUE;
		if (entry->selected)
		{
			if (entry->length != -1)
				selection += entry->length;
			else
				selection_more = TRUE;
		}
		list = g_list_next(list);
	}
	PL_UNLOCK();
	selection /= 1000;

	if (selection > 0 || (selection == 0 && !selection_more))
	{
		if (selection > 3600)
			sel_text = g_strdup_printf("%d:%-2.2d:%-2.2d%s", selection / 3600, (selection / 60) % 60, selection % 60, (selection_more ? "+" : ""));
		else
			sel_text = g_strdup_printf("%d:%-2.2d%s", selection / 60, selection % 60, (selection_more ? "+" : ""));
	}
	else
		sel_text = g_strdup("?");
	total /= 1000;
	if (total > 0 || (total == 0 && !total_more))
	{
		if (total > 3600)
			tot_text = g_strdup_printf("%d:%-2.2d:%-2.2d%s", total / 3600, (total / 60) % 60, total % 60, total_more ? "+" : "");
		else
			tot_text = g_strdup_printf("%d:%-2.2d%s", total / 60, total % 60, total_more ? "+" : "");
	}
	else
		tot_text = g_strdup("?");
	text = g_strconcat(sel_text, "/", tot_text, NULL);
	textbox_set_text(playlistwin_info, text);
	g_free(text);
	g_free(tot_text);
	g_free(sel_text);
}

void playlistwin_update_sinfo(void)
{
	gchar *time = NULL, *pos, *title, *info, *tmp, *tmp2;
	gint max_len, i;
	PlaylistEntry *entry;
	GList *playlist;

	PL_LOCK();
	playlist = get_playlist();

	if (!playlist)
	{
		PL_UNLOCK();
		textbox_set_text(playlistwin_sinfo, "");
		return;
	}

	entry = g_list_nth(playlist, __get_playlist_position())->data;

	title = entry->title;
	if (!title)
		title = g_basename(entry->filename);

	max_len = (cfg.playlist_width - 35) / 5;

	pos = g_strdup_printf("%d. ", __get_playlist_position() + 1);

	if (entry->length != -1)
	{
		time = g_strdup_printf("%d:%-2.2d", entry->length / 60000, (entry->length / 1000) % 60);
		max_len -= strlen(time);
		info = g_strdup_printf("%s%-*.*s%s", pos, max_len - (int)strlen(pos), max_len - (int)strlen(pos), title, time);
		g_free(time);

	}
	else
		info = g_strdup_printf("%s%-*.*s", pos, max_len - (int)strlen(pos), max_len - (int)strlen(pos), title);
	PL_UNLOCK();
	g_free(pos);
	if (strlen(title) > max_len - (time ? 1 : 0))
	{
		for (i = 0; i < 3; i++)
			info[max_len - i - (time ? 2 : 1)] = '.';
		if (time)
			info[max_len - 1] = ' ';
	}

	if (cfg.convert_underscore)
		while ((tmp = strchr(info, '_')) != NULL)
			*tmp = ' ';
	if (cfg.convert_twenty)
	{
		while ((tmp = strstr(info, "%20")) != NULL)
		{
			tmp2 = tmp + 3;
			*(tmp++) = ' ';
			while (*tmp2)
				*(tmp++) = *(tmp2++);
			*tmp = '\0';
		}
	}

	textbox_set_text(playlistwin_sinfo, info);
	g_free(info);
}

gboolean playlistwin_item_visible(gint index)
{
	if (index >= playlistwin_list->pl_first && index < (playlistwin_list->pl_first + playlistwin_list->pl_num_visible))
		return TRUE;
	return FALSE;
}

gint playlistwin_get_toprow(void)
{
	if (playlistwin_list)
		return (playlistwin_list->pl_first);
	return (-1);
}

void playlistwin_set_toprow(gint toprow)
{
	if (playlistwin_list)
		playlistwin_list->pl_first = toprow;
	playlistwin_update_list();
}

void playlistwin_update_list(void)
{
	if (playlistwin_list)
		draw_widget(playlistwin_list);
	if (playlistwin_slider)
		draw_widget(playlistwin_slider);
	if (playlistwin_info)
		playlistwin_update_info();
	if (playlistwin_sinfo)
		playlistwin_update_sinfo();
}

static void playlistwin_create_mask(void)
{
	GdkBitmap *tmp;
	GdkGC *gc;
	GdkColor pattern;

	tmp = playlistwin_mask;
	playlistwin_mask = gdk_pixmap_new(playlistwin->window, cfg.playlist_width, cfg.playlist_shaded ? 14 : cfg.playlist_height, 1);
	gc = gdk_gc_new(playlistwin_mask);
	pattern.pixel = 1;
	gdk_gc_set_foreground(gc, &pattern);
	gdk_draw_rectangle(playlistwin_mask, gc, TRUE, 0, 0, cfg.playlist_width, cfg.playlist_shaded ? 14 : cfg.playlist_height);
	gdk_gc_destroy(gc);
	gtk_widget_shape_combine_mask(playlistwin, playlistwin_mask, 0, 0);

	if (tmp)
		gdk_bitmap_unref(tmp);
}

void playlistwin_set_shade(gboolean shaded)
{
	if (shaded)
	{
		dock_shade(dock_window_list, playlistwin, 14);
		show_widget(playlistwin_sinfo);
		playlistwin_shade->pb_nx = 128;
		playlistwin_shade->pb_ny = 45;
		playlistwin_shade->pb_px = 150;
		playlistwin_shade->pb_py = 42;
		playlistwin_close->pb_nx = 138;
		playlistwin_close->pb_ny = 45;
	}
	else
	{
		dock_shade(dock_window_list, playlistwin, cfg.playlist_height);
		hide_widget(playlistwin_sinfo);
		playlistwin_shade->pb_nx = 157;
		playlistwin_shade->pb_ny = 3;
		playlistwin_shade->pb_px = 62;
		playlistwin_shade->pb_py = 42;
		playlistwin_close->pb_nx = 167;
		playlistwin_close->pb_ny = 3;
	}
	cfg.playlist_shaded = shaded;
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(mainwin_options_menu, _("/Playlist WindowShade Mode")))->active = shaded;

	playlistwin_create_mask();
	draw_playlist_window(TRUE);
}

void playlistwin_shade_cb(void)
{
	playlistwin_set_shade(!cfg.playlist_shaded);
}

void playlistwin_raise(void)
{
	if (cfg.playlist_visible)
		gdk_window_raise(playlistwin->window);
}

void playlistwin_move(gint x, gint y)
{
	cfg.playlist_x = x;
	cfg.playlist_y = y;
	if (cfg.playlist_visible)
		gdk_window_move(playlistwin->window, x, y);
}

void playlistwin_release(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
	gdk_pointer_ungrab(GDK_CURRENT_TIME);
	gdk_flush();
	if (playlistwin_resizing)
	{
		playlistwin_resizing = FALSE;
	}
	else if (dock_is_moving(playlistwin))
	{
		dock_move_release(playlistwin);
	}
	else
	{
		handle_release_cb(playlistwin_wlist, widget, event);
		playlist_popup_destroy();
		draw_playlist_window(FALSE);
	}
}

void playlistwin_scroll_up_pushed(void)
{
	playlistwin_list->pl_first -= 3;
	playlistwin_update_list();
}

void playlistwin_scroll_down_pushed(void)
{
	playlistwin_list->pl_first += 3;
	playlistwin_update_list();
}

void playlistwin_select_all(void)
{
	GList *node;

	PL_LOCK();
	node = get_playlist();
	while (node)
	{
		((PlaylistEntry *) node->data)->selected = TRUE;
		node = node->next;
	}
	PL_UNLOCK();
	playlistwin_list->pl_prev_selected=0;
	playlistwin_list->pl_prev_min=0;
	playlistwin_list->pl_prev_max=get_playlist_length()-1;
	playlistwin_update_list();
}

void playlistwin_select_none(void)
{
	GList *node;

	PL_LOCK();
	node = get_playlist();
	while (node)
	{
		((PlaylistEntry *) node->data)->selected = FALSE;
		node = node->next;
	}
	PL_UNLOCK();
	playlistwin_list->pl_prev_selected=-1;
	playlistwin_list->pl_prev_min=-1;
	playlistwin_update_list();
}

void playlistwin_inverse_selection(void)
{
	GList *node;

	PL_LOCK();
	node = get_playlist();
	while (node)
	{
		((PlaylistEntry *) node->data)->selected = !((PlaylistEntry *) node->data)->selected;
		node = node->next;
	}
	PL_UNLOCK();
	playlistwin_list->pl_prev_selected=-1;
	playlistwin_list->pl_prev_min=-1;
	playlistwin_update_list();
}

void playlistwin_motion(GtkWidget * widget, GdkEventMotion * event, gpointer callback_data)
{
	gint bx, by, w, h, nw, nh;
	XEvent ev;
	GdkPixmap *oldbg;
	gboolean dummy;

	if (playlistwin_resizing)
	{
		w = event->x + playlistwin_resize_x;
		bx = (w - 275) / 25;
		nw = (bx * 25) + 275;
		if (nw < 275)
			nw = 275;

		if (!cfg.playlist_shaded)
		{
			h = event->y + playlistwin_resize_y;
			by = (h - 58) / 29;
			nh = (by * 29) + 58;
			if (nh < 116)
				nh = 116;
		}
		else
			nh = cfg.playlist_height;

		if (nw != cfg.playlist_width || nh != cfg.playlist_height)
		{
			cfg.playlist_width = nw;
			cfg.playlist_height = nh;
			resize_widget(playlistwin_list, cfg.playlist_width - 31, cfg.playlist_height - 58);
			move_widget(playlistwin_slider, cfg.playlist_width - 15, 20);
			resize_widget(playlistwin_sinfo, cfg.playlist_width - 35, 14);
			playlistwin_update_sinfo();
			move_widget(playlistwin_shade, cfg.playlist_width - 21, 3);
			move_widget(playlistwin_close, cfg.playlist_width - 11, 3);
			move_widget(playlistwin_time_min, cfg.playlist_width - 82, cfg.playlist_height - 15);
			move_widget(playlistwin_time_sec, cfg.playlist_width - 64, cfg.playlist_height - 15);
			move_widget(playlistwin_info, cfg.playlist_width - 143, cfg.playlist_height - 28);
			move_widget(playlistwin_srew, cfg.playlist_width - 144, cfg.playlist_height - 16);
			move_widget(playlistwin_splay, cfg.playlist_width - 138, cfg.playlist_height - 16);
			move_widget(playlistwin_spause, cfg.playlist_width - 128, cfg.playlist_height - 16);
			move_widget(playlistwin_sstop, cfg.playlist_width - 118, cfg.playlist_height - 16);
			move_widget(playlistwin_sfwd, cfg.playlist_width - 109, cfg.playlist_height - 16);
			move_widget(playlistwin_seject, cfg.playlist_width - 100, cfg.playlist_height - 16);
			move_widget(playlistwin_sscroll_up, cfg.playlist_width - 14, cfg.playlist_height - 35);
			move_widget(playlistwin_sscroll_down, cfg.playlist_width - 14, cfg.playlist_height - 30);
			resize_widget(playlistwin_slider, 8, cfg.playlist_height - 58);
			if (cfg.playlist_width >= 350)
			{
				move_widget(playlistwin_vis, cfg.playlist_width - 223, cfg.playlist_height - 26);
				if (playlistwin_vis_enabled)
					show_widget(playlistwin_vis);
			}
			else
			{
				hide_widget(playlistwin_vis);
			}
			oldbg = playlistwin_bg;
			playlistwin_bg = gdk_pixmap_new(playlistwin->window, cfg.playlist_width, cfg.playlist_height,
							gdk_rgb_get_visual()->depth);
			widget_list_change_pixmap(playlistwin_wlist, playlistwin_bg);
			playlistwin_create_mask();

			playlistwin_draw_frame();
			draw_widget_list(playlistwin_wlist, &dummy, TRUE);
			clear_widget_list_redraw(playlistwin_wlist);
			gdk_window_set_back_pixmap(playlistwin->window, playlistwin_bg, 0);
			if (!cfg.playlist_shaded)
			{
				gdk_window_set_hints(playlistwin->window, 0, 0, cfg.playlist_width, cfg.playlist_height, cfg.playlist_width, cfg.playlist_height, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
				gdk_window_resize(playlistwin->window, cfg.playlist_width, cfg.playlist_height);
				gtk_widget_set_usize(playlistwin, cfg.playlist_width, cfg.playlist_height);
			}
			else
			{
				gdk_window_set_hints(playlistwin->window, 0, 0, cfg.playlist_width, 14, cfg.playlist_width, 14, GDK_HINT_MIN_SIZE | GDK_HINT_MAX_SIZE);
				gdk_window_resize(playlistwin->window, cfg.playlist_width, 14);
				gtk_widget_set_usize(playlistwin, cfg.playlist_width, 14);
			}
			gdk_window_clear(playlistwin->window);
			gdk_pixmap_unref(oldbg);
		}

	}
	else if (dock_is_moving(playlistwin))
	{
		dock_move_motion(playlistwin, event);
	}
	else
	{
		handle_motion_cb(playlistwin_wlist, widget, event);
		draw_playlist_window(FALSE);
	}
	gdk_flush();
	while (XCheckMaskEvent(GDK_DISPLAY(), ButtonMotionMask, &ev)) ;
}

void playlistwin_show_filebrowser(void)
{
	static GtkWidget *filebrowser;
	if (filebrowser != NULL)
		return;

	filebrowser = util_create_filebrowser(FALSE);
	gtk_signal_connect(GTK_OBJECT(filebrowser), "destroy",
			   GTK_SIGNAL_FUNC(gtk_widget_destroyed), &filebrowser);
}



void playlistwin_url_ok_clicked(GtkWidget * w, GtkWidget * entry)
{
	gchar *text, *temp;

	text = gtk_entry_get_text(GTK_ENTRY(entry));
	if (text && *text)
	{
		g_strstrip(text);
		if(strstr(text, ":/") == NULL && text[0] != '/')
			temp = g_strconcat("http://", text, NULL);
		else
			temp = g_strdup(text);
		playlist_add_url_string(temp);
		g_free(temp);
		playlistwin_update_list();
	}
	gtk_widget_destroy(playlistwin_url_window);
}

void playlistwin_show_add_url_window(void)
{
	if(!playlistwin_url_window)
	{
		playlistwin_url_window = util_create_add_url_window(_("Enter URL to add:"), GTK_SIGNAL_FUNC(playlistwin_url_ok_clicked), NULL);
		gtk_window_set_transient_for(GTK_WINDOW(playlistwin_url_window), GTK_WINDOW(playlistwin));
		gtk_signal_connect(GTK_OBJECT(playlistwin_url_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &playlistwin_url_window);
		gtk_widget_show(playlistwin_url_window);
	}

}

void playlistwin_add_dir_handler(gchar * dir)
{
	g_free(cfg.filesel_path);
	cfg.filesel_path = g_strdup(dir);
	playlist_add_dir(dir);
	playlistwin_update_list();
}

void playlistwin_show_dirbrowser(void)
{
	if (!playlistwin_dir_browser)
	{
		playlistwin_dir_browser = xmms_create_dir_browser(_("Select directory to add:"), cfg.filesel_path, GTK_SELECTION_EXTENDED, playlistwin_add_dir_handler);
		gtk_signal_connect(GTK_OBJECT(playlistwin_dir_browser), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &playlistwin_dir_browser);
		gtk_window_set_transient_for(GTK_WINDOW(playlistwin_dir_browser), GTK_WINDOW(playlistwin));
	}
	if (!GTK_WIDGET_VISIBLE(playlistwin_dir_browser))
		gtk_widget_show(playlistwin_dir_browser);

}

void playlistwin_add_popup_handler(gint item)
{
	switch (item)
	{
		case ADD_URL:
			playlistwin_show_add_url_window();
			break;
		case ADD_DIR:
			playlistwin_show_dirbrowser();
			break;
		case ADD_FILE:
			playlistwin_show_filebrowser();
			break;
	}
}

void playlistwin_sub_popup_handler(gint item)
{
	gint x, y;
	
	switch (item)
	{
		case SUB_MISC:
			gdk_window_get_pointer(NULL, &x, &y, NULL);
			util_item_factory_popup(GTK_ITEM_FACTORY(playlistwin_sub_menu), x, y, 1, GDK_CURRENT_TIME);
			break;
		case SUB_ALL:
			playlist_clear();
			playlistwin_update_list();
			mainwin_set_song_info(0, 0, 0);
			mainwin_set_info_text();
			break;
		case SUB_CROP:
			playlist_delete(TRUE);
			break;
		case SUB_SELECTED:
			playlist_delete(FALSE);
			break;
	}
}

void playlistwin_sel_popup_handler(gint item)
{
	switch (item)
	{
		case SEL_INV:
			playlistwin_inverse_selection();
			break;
		case SEL_ZERO:
			playlistwin_select_none();
			break;
		case SEL_ALL:
			playlistwin_select_all();
			break;
	}
}

static void playlistwin_set_sensitive_sortmenu(gboolean set)
{
	gtk_widget_set_sensitive(gtk_item_factory_get_widget(playlistwin_sort_menu, _("/Sort Selection/By Title")),set);
	gtk_widget_set_sensitive(gtk_item_factory_get_widget(playlistwin_sort_menu, _("/Sort Selection/By Filename")),set);
	gtk_widget_set_sensitive(gtk_item_factory_get_widget(playlistwin_sort_menu, _("/Sort Selection/By Path + Filename")),set);

#if 0
	gtk_widget_set_sensitive(gtk_item_factory_get_widget(playlistwin_sort_menu, _("/Sort Selection")),set);
#endif

}

void playlistwin_misc_popup_handler(gint item)
{
	gint x, y, i;
	GList *node;
	GList *list;
	gchar *filename = NULL;

	switch (item)
	{
		case MISC_SORT:
			PL_LOCK();
			list = get_playlist();
			i = 0;
			while(list)
			{
				if(((PlaylistEntry *) list->data)->selected)
				{
					i++;
					if (i > 1)
						break;
				}
				list = g_list_next(list);
			}
			PL_UNLOCK();
			if (i > 1)
				playlistwin_set_sensitive_sortmenu(TRUE);
			else
				playlistwin_set_sensitive_sortmenu(FALSE);
				
			gdk_window_get_pointer(NULL, &x, &y, NULL);
			util_item_factory_popup(GTK_ITEM_FACTORY(playlistwin_sort_menu), x, y, 1, GDK_CURRENT_TIME);
			break;
		case MISC_FILEINFO:
			PL_LOCK();
			node = get_playlist();
			while (node)
			{
				if (((PlaylistEntry *) node->data)->selected)
				{
					filename = g_strdup(((PlaylistEntry *) node->data)->filename);
					break;
				}
				node = node->next;
			}
			PL_UNLOCK();
			if(filename)
			{
				input_file_info_box(filename);
				g_free(filename);
			}
			break;
		case MISC_MISCOPTS:
			break;

	}
}

void playlistwin_save_filesel_ok(GtkWidget * w, GtkWidget * filesel)
{
	gchar *filename, *text, *tmp;

	if (util_filebrowser_is_dir(GTK_FILE_SELECTION(filesel)))
		return;
	
	filename = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
	text = g_strdup(filename);

	if ((tmp = strrchr(text, '/')) != NULL)
		*(tmp + 1) = '\0';
	g_free(cfg.playlist_path);
	cfg.playlist_path = g_strdup(text);
	g_free(text);

	if (filename && *filename)
		playlist_save(filename);
	gtk_widget_destroy(GTK_WIDGET(filesel));
}

void playlistwin_load_filesel_ok(GtkWidget * w, GtkWidget * filesel)
{
	gchar *filename, *text, *tmp;

	if (util_filebrowser_is_dir(GTK_FILE_SELECTION(filesel)))
		return;
	
	filename = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
	text = g_strdup(filename);

	if ((tmp = strrchr(text, '/')) != NULL)
		*(tmp + 1) = '\0';
	g_free(cfg.playlist_path);
	cfg.playlist_path = g_strdup(text);
	g_free(text);

	if (filename && *filename)
	{
		playlist_clear();
		playlistwin_update_list();
		mainwin_set_song_info(0, 0, 0);
		mainwin_set_info_text();

		playlist_load(filename);
		playlistwin_update_list();
	}
	gtk_widget_destroy(GTK_WIDGET(filesel));
}

void playlistwin_plist_popup_handler(gint item)
{
	switch (item)
	{
		case PLIST_NEW:
			playlist_clear();
			playlistwin_update_list();
			mainwin_set_song_info(0, 0, 0);
			mainwin_set_info_text();
			break;

		case PLIST_SAVE:
			if (playlistwin_save_filesel != NULL)
				break;

			playlistwin_save_filesel = gtk_file_selection_new(_("Save playlist"));
			if (cfg.playlist_path)
				gtk_file_selection_set_filename(GTK_FILE_SELECTION(playlistwin_save_filesel), cfg.playlist_path);
			gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(playlistwin_save_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(playlistwin_save_filesel_ok), playlistwin_save_filesel);
			gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(playlistwin_save_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(playlistwin_save_filesel));
			gtk_signal_connect(GTK_OBJECT(playlistwin_save_filesel),
					   "destroy",
					   GTK_SIGNAL_FUNC(gtk_widget_destroyed),
					   &playlistwin_save_filesel);
			gtk_widget_show(playlistwin_save_filesel);
			break;

		case PLIST_LOAD:
			if (playlistwin_load_filesel != NULL)
				break;
			playlistwin_load_filesel = gtk_file_selection_new(_("Load playlist"));
			if (cfg.playlist_path)
				gtk_file_selection_set_filename(GTK_FILE_SELECTION(playlistwin_load_filesel), cfg.playlist_path);
			gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(playlistwin_load_filesel)->ok_button), "clicked", GTK_SIGNAL_FUNC(playlistwin_load_filesel_ok), playlistwin_load_filesel);
			gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(playlistwin_load_filesel)->cancel_button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(playlistwin_load_filesel));
			gtk_signal_connect(GTK_OBJECT(playlistwin_load_filesel),
					   "destroy",
					   GTK_SIGNAL_FUNC(gtk_widget_destroyed),
					   &playlistwin_load_filesel);
			gtk_widget_show(playlistwin_load_filesel);
			break;
	}
}

static gboolean inside_sensitive_widgets(gint x, gint y)
{
	return (inside_widget(x, y, playlistwin_list) ||
		inside_widget(x, y, playlistwin_slider) ||
		inside_widget(x, y, playlistwin_close) ||
		inside_widget(x, y, playlistwin_shade) ||
		inside_widget(x, y, playlistwin_time_min) ||
		inside_widget(x, y, playlistwin_time_sec) ||
		inside_widget(x, y, playlistwin_info) ||
		inside_widget(x, y, playlistwin_vis) ||
		inside_widget(x, y, playlistwin_srew) ||
		inside_widget(x, y, playlistwin_splay) ||
		inside_widget(x, y, playlistwin_spause) ||
		inside_widget(x, y, playlistwin_sstop) ||
		inside_widget(x, y, playlistwin_sfwd) ||
		inside_widget(x, y, playlistwin_seject) ||
		inside_widget(x, y, playlistwin_sscroll_up) ||
		inside_widget(x, y, playlistwin_sscroll_down));
}

#define REGION_L(x1,x2,y1,y2) (event->x >= (x1) && event->x < (x2) && \
			       event->y >= cfg.playlist_height - (y1) && \
			       event->y < cfg.playlist_height - (y2))
#define REGION_R(x1,x2,y1,y2) (event->x >= cfg.playlist_width - (x1) && \
			       event->x < cfg.playlist_width - (x2) && \
			       event->y >= cfg.playlist_height - (y1) && \
			       event->y < cfg.playlist_height - (y2))

void playlistwin_press(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
	gint add_nx[] =	{0, 0, 0},
	add_ny[] = {111, 130, 149},
	add_sx[] = {23, 23, 23},
	add_sy[] = {111, 130, 149},
	add_barx = 48, add_bary = 111;
	
	gint sub_nx[] =	{54, 54, 54, 54},
	sub_ny[] = {168, 111, 130, 149},
	sub_sx[] = {77, 77, 77, 77},
	sub_sy[] = {168, 111, 130, 149},
	sub_barx = 100, sub_bary = 111;
	
	gint sel_nx[] =	{104, 104, 104},
	sel_ny[] = {111, 130, 149},
	sel_sx[] = {127, 127, 127},
	sel_sy[] = {111, 130, 149},
	sel_barx = 150, sel_bary = 111;
	
	gint misc_nx[] = {154, 154, 154},
	misc_ny[] = {111, 130, 149},
	misc_sx[] = {177, 177, 177},
	misc_sy[] = {111, 130, 149},
	misc_barx = 200, misc_bary = 111;
	
	gint plist_nx[] = {204, 204, 204},
	plist_ny[] = {111, 130, 149},
	plist_sx[] = {227, 227, 227},
	plist_sy[] = {111, 130, 149},
	plist_barx = 250, plist_bary = 111;

	gboolean grab = TRUE;

	if (event->button == 1 &&
	    ((!cfg.playlist_shaded && event->x > cfg.playlist_width - 20 &&
	      event->y > cfg.playlist_height - 20) ||
	     (cfg.playlist_shaded &&
	      event->x >= cfg.playlist_width - 31 &&
	      event->x < cfg.playlist_width - 22)))
	{
		playlistwin_resizing = TRUE;
		playlistwin_resize_x = cfg.playlist_width - event->x;
		playlistwin_resize_y = cfg.playlist_height - event->y;
		playlistwin_raise();
	}
	else if (event->button == 1 && REGION_L(12, 37, 29, 11))
	{
		dock_get_widget_pos(playlistwin, &cfg.playlist_x, &cfg.playlist_y);
		playlist_popup(cfg.playlist_x + 12,
			       cfg.playlist_y + cfg.playlist_height - (3 * 18) - 11,
			       3, add_nx, add_ny, add_sx, add_sy, add_barx, add_bary,
			       playlistwin_add_popup_handler);
		grab = FALSE;
	}
	else if (event->button == 1 && REGION_L(41, 66, 29, 11))
	{
		dock_get_widget_pos(playlistwin, &cfg.playlist_x, &cfg.playlist_y);
		playlist_popup(cfg.playlist_x + 41,
			       cfg.playlist_y + cfg.playlist_height - (4 * 18) - 11,
			       4, sub_nx, sub_ny, sub_sx, sub_sy, sub_barx, sub_bary,
			       playlistwin_sub_popup_handler);
		grab = FALSE;
	}
	else if (event->button == 1 && REGION_L(70, 95, 29, 11))
	{
		dock_get_widget_pos(playlistwin, &cfg.playlist_x, &cfg.playlist_y);
		playlist_popup(cfg.playlist_x + 70,
			       cfg.playlist_y + cfg.playlist_height - (3 * 18) - 11,
			       3, sel_nx, sel_ny, sel_sx, sel_sy, sel_barx, sel_bary,
			       playlistwin_sel_popup_handler);
		grab = FALSE;
	}
	else if (event->button == 1 && REGION_L(99, 124, 29, 11))
	{
		dock_get_widget_pos(playlistwin, &cfg.playlist_x, &cfg.playlist_y);
		playlist_popup(cfg.playlist_x + 99,
			       cfg.playlist_y + cfg.playlist_height - (3 * 18) - 11,
			       3, misc_nx, misc_ny, misc_sx, misc_sy, misc_barx,
			       misc_bary, playlistwin_misc_popup_handler);
		grab = FALSE;
	}
	else if (event->button == 1 && REGION_R(46, 23, 29, 11))
	{
		dock_get_widget_pos(playlistwin, &cfg.playlist_x, &cfg.playlist_y);
		playlist_popup(cfg.playlist_x + cfg.playlist_width - 46,
			       cfg.playlist_y + cfg.playlist_height - (3 * 18) - 11,
			       3, plist_nx, plist_ny, plist_sx, plist_sy, plist_barx,
			       plist_bary, playlistwin_plist_popup_handler);
		grab = FALSE;
	}
	else if (event->button == 1 && REGION_R(82, 54, 15, 9))
	{
		if (cfg.timer_mode == TIMER_ELAPSED)
			cfg.timer_mode = TIMER_REMAINING;
		else
			cfg.timer_mode = TIMER_ELAPSED;
	}
	else if (cfg.playlist_width >= 350 && REGION_R(223, 151, 26, 10))
	{
		if (event->button == 1)
		{
			cfg.vis_type++;
			if (cfg.vis_type > VIS_OFF)
				cfg.vis_type = VIS_ANALYZER;
			mainwin_vis_set_type(cfg.vis_type);
		}
		else if (event->button == 3)
		{
			gint mx, my;
			GdkModifierType modmask;

			gdk_window_get_pointer(NULL, &mx, &my, &modmask);
			util_item_factory_popup(mainwin_vis_menu, mx, my, 3, GDK_CURRENT_TIME);
			grab = FALSE;
		}
	}
	else if (event->button == 1 && event->type == GDK_BUTTON_PRESS &&
		 !inside_sensitive_widgets(event->x, event->y) &&
		 (cfg.easy_move || event->y < 14))
	{
		gdk_window_raise(playlistwin->window);
		dock_move_press(dock_window_list, playlistwin, event, FALSE);

	}
	else if (event->button == 1 && event->type == GDK_2BUTTON_PRESS &&
		 !inside_sensitive_widgets(event->x, event->y) && event->y < 14)
	{
		playlistwin_set_shade(!cfg.playlist_shaded);
		if(dock_is_moving(playlistwin))
			dock_move_release(playlistwin);
	}
	else if (event->button == 3 &&
		 !(inside_widget(event->x, event->y, playlistwin_list) ||
		   (event->y >= cfg.playlist_height - 29 &&
		    event->y < cfg.playlist_height - 11 &&
		    ((event->x >= 12 && event->x < 37) ||
		     (event->x >= 41 && event->x < 66) ||
		     (event->x >= 70 && event->x < 95) ||
		     (event->x >= 99 && event->x < 124) ||
		     (event->x >= cfg.playlist_width - 46 &&
		      event->x < cfg.playlist_width - 23)))))
	{
		/*
		 * Pop up the main menu a few pixels down to avoid
		 * anything to be selected initially.
		 */
		util_item_factory_popup(mainwin_general_menu, event->x_root, event->y_root + 2, 3, GDK_CURRENT_TIME);
		grab = FALSE;
	}
	else if (event->button == 4)
		playlistwin_scroll_up_pushed();
	else if(event->button == 5)
		playlistwin_scroll_down_pushed();
	else
	{
		handle_press_cb(playlistwin_wlist, widget, event);
		draw_playlist_window(FALSE);
	}
	if (grab)
		gdk_pointer_grab(playlistwin->window, FALSE,
				 GDK_BUTTON_MOTION_MASK | GDK_BUTTON_RELEASE_MASK,
				 GDK_NONE, GDK_NONE, GDK_CURRENT_TIME);

}

void playlistwin_focus_in(GtkWidget * widget, GdkEvent * event, gpointer callback_data)
{
	playlistwin_close->pb_allow_draw = TRUE;
	playlistwin_shade->pb_allow_draw = TRUE;
	playlistwin_focus = TRUE;
	draw_playlist_window(TRUE);
}

void playlistwin_focus_out(GtkWidget * widget, GdkEventButton * event, gpointer callback_data)
{
	playlistwin_close->pb_allow_draw = FALSE;
	playlistwin_shade->pb_allow_draw = FALSE;
	playlistwin_focus = FALSE;
	draw_playlist_window(TRUE);
}

static gboolean playlistwin_configure(GtkWidget * window, GdkEventConfigure *event, gpointer data)
{
	if (!GTK_WIDGET_VISIBLE(window))
		return FALSE;
	
	gdk_window_get_deskrelative_origin(window->window, &cfg.playlist_x, &cfg.playlist_y);
	return FALSE;
}

void playlistwin_set_back_pixmap()
{
	gdk_window_set_back_pixmap(playlistwin->window, playlistwin_bg, 0);
	gdk_window_clear(playlistwin->window);
}

gint playlistwin_client_event(GtkWidget *w,GdkEventClient *event, gpointer data)
{
	static GdkAtom atom_rcfiles = GDK_NONE;

	if (!atom_rcfiles)
		atom_rcfiles = gdk_atom_intern("_GTK_READ_RCFILES", FALSE);
	if(event->message_type == atom_rcfiles)
	{
		mainwin_set_back_pixmap();
		equalizerwin_set_back_pixmap();
		playlistwin_set_back_pixmap();
		return TRUE;
		
	}
	return FALSE;
}

gint playlistwin_delete(GtkWidget * w, gpointer data)
{
	playlistwin_hide();
	return TRUE;
}

gboolean playlistwin_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	GList *list;
	PlaylistEntry *entry;
	guint keyval;
	gboolean refresh = TRUE;

	if (cfg.playlist_shaded)
	{
		gtk_widget_event(mainwin, (GdkEvent *) event);
		return TRUE;
	}
	
	switch (keyval = event->keyval)
	{
		case GDK_KP_Up:
		case GDK_KP_Down:
			keyval = event->keyval == GDK_KP_Up ? GDK_Up : GDK_Down;
		case GDK_Up:
		case GDK_Down:
			if ((event->state & GDK_MOD1_MASK) && (event->state & GDK_SHIFT_MASK))
				break;
			if (event->state & GDK_MOD1_MASK)
			{
				if (playlistwin_list->pl_prev_selected > -1)
				{
					if (keyval == GDK_Up)
						playlist_list_move_up();
					else
						playlist_list_move_down();
				}
			}
			else
			{
				PL_LOCK();
				list = get_playlist();
				while (list)
				{
					entry = (PlaylistEntry *) list->data;
					entry->selected = FALSE;
					list = list->next;
				}
				PL_UNLOCK();
			}

			if (playlistwin_list->pl_prev_selected == -1 ||
			    (!playlistwin_item_visible(playlistwin_list->pl_prev_selected) &&
			     !(event->state & GDK_SHIFT_MASK && playlistwin_list->pl_prev_min != -1)))
			{
				playlistwin_list->pl_prev_selected = playlistwin_list->pl_first;
			}
			else if (event->state & GDK_SHIFT_MASK)
			{
				if (playlistwin_list->pl_prev_min == -1)
				{
					playlistwin_list->pl_prev_max = playlistwin_list->pl_prev_selected;
					playlistwin_list->pl_prev_min = playlistwin_list->pl_prev_selected;
				}
				playlistwin_list->pl_prev_max += (keyval == GDK_Up ? -1 : 1);
				playlistwin_list->pl_prev_max = CLAMP(playlistwin_list->pl_prev_max, 0, get_playlist_length() - 1);

				playlistwin_list->pl_first = MIN(playlistwin_list->pl_first, playlistwin_list->pl_prev_max);
				playlistwin_list->pl_first = MAX(playlistwin_list->pl_first, playlistwin_list->pl_prev_max - playlistwin_list->pl_num_visible + 1);
				playlist_list_select_range(playlistwin_list->pl_prev_min, playlistwin_list->pl_prev_max, TRUE);
				break;
			}
			else if (keyval == GDK_Up)
				playlistwin_list->pl_prev_selected--;
			else
				playlistwin_list->pl_prev_selected++;

			playlistwin_list->pl_prev_selected = CLAMP(playlistwin_list->pl_prev_selected, 0, get_playlist_length() - 1);

			if (playlistwin_list->pl_prev_selected < playlistwin_list->pl_first)
				playlistwin_list->pl_first--;
			else if (playlistwin_list->pl_prev_selected >= (playlistwin_list->pl_first + playlistwin_list->pl_num_visible))
				playlistwin_list->pl_first++;

			if (event->state & GDK_MOD1_MASK)
				break;
			if ((list = g_list_nth(get_playlist(), playlistwin_list->pl_prev_selected)) != NULL)
			{
				entry = list->data;
				entry->selected = TRUE;
				playlistwin_list->pl_prev_min = -1;
			}
			break;
		case GDK_Page_Up:
			playlistwin_scroll_up_pushed();
			break;
		case GDK_Page_Down:
			playlistwin_scroll_down_pushed();
			break;
		case GDK_Home:
			playlistwin_list->pl_first = 0;
			break;
		case GDK_End:
			playlistwin_list->pl_first = get_playlist_length() - playlistwin_list->pl_num_visible;
			break;
		case GDK_Return:
			if (playlistwin_list->pl_prev_selected > -1 && playlistwin_item_visible(playlistwin_list->pl_prev_selected))
			{
				playlist_set_position(playlistwin_list->pl_prev_selected);
				if (!get_input_playing())
					playlist_play();
			}
			refresh = FALSE;
			break;
		case GDK_3:
			if (event->state & GDK_CONTROL_MASK)
			{
				if (playlistwin_list->pl_prev_selected > -1 && playlistwin_item_visible(playlistwin_list->pl_prev_selected))
					playlist_fileinfo(playlistwin_list->pl_prev_selected);
				else
					playlist_fileinfo_current();
			}
			refresh = FALSE;
			break;
		case GDK_Delete:
			if (event->state & GDK_SHIFT_MASK)
				playlist_delete(TRUE);
			else
				playlist_delete(FALSE);
			refresh = FALSE;
			break;
		case GDK_Insert:
			if(event->state & GDK_SHIFT_MASK)
				playlistwin_show_dirbrowser();
			else if(event->state & GDK_MOD1_MASK)
				playlistwin_show_add_url_window();
			else
				playlistwin_show_filebrowser();
			refresh=FALSE;
			break;
		default:
			gtk_widget_event(mainwin, (GdkEvent *) event);
			refresh = FALSE;
			break;
	}
	if (refresh)
		playlistwin_update_list();

	return TRUE;
}

void playlistwin_draw_frame(void)
{
	gint w, h, y, i, c;
	SkinIndex src;

	w = cfg.playlist_width;
	h = cfg.playlist_height;
	src = SKIN_PLEDIT;

	if (cfg.playlist_shaded)
	{
		skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 72, 42, 0, 0, 25, 14);
		c = (w - 75) / 25;
		for (i = 0; i < c; i++)
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 72, 57, (i * 25) + 25, 0, 25, 14);
		skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 99, (!playlistwin_focus && cfg.dim_titlebar) ? 57 : 42, w - 50, 0, 50, 14);
	}
	else
	{
		if (playlistwin_focus || !cfg.dim_titlebar)
			y = 0;
		else
			y = 21;
		/* Titlebar left corner */
		skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 0, y, 0, 0, 25, 20);
		c = (w - 150) / 25;
		/* Titlebar, left and right of the title */
		for (i = 0; i < c / 2; i++)
		{
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 127, y, (i * 25) + 25, 0, 25, 20);
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 127, y, (i * 25) + (w / 2) + 50, 0, 25, 20);
		}
		if (c & 1)
		{
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 127, y, ((c / 2) * 25) + 25, 0, 12, 20);
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 127, y, (w / 2) + ((c / 2) * 25) + 50, 0, 13, 20);
		}

		/* Titlebar title */
		skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 26, y, (w / 2) - 50, 0, 100, 20);
		/* Titlebar, right corner */
		skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 153, y, w - 25, 0, 25, 20);

		/* Left and right side */
		for (i = 0; i < (h - 58) / 29; i++)
		{
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 0, 42, 0, (i * 29) + 20, 12, 29);
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 32, 42, w - 19, (i * 29) + 20, 19, 29);
		}
		/* Bottom left corner (menu buttons) */
		skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 0, 72, 0, h - 38, 125, 38);
		c = (w - 275) / 25;
		/* Visualization window */
		if (c >= 3)
		{
			c -= 3;
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 205, 0, w - 225, h - 38, 75, 38);
		}
		/* Bottom blank parts */
		for (i = 0; i < c; i++)
			skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 179, 0, (i * 25) + 125, h - 38, 25, 38);
		/* Bottom right corner (playbuttons etc) */
		skin_draw_pixmap(playlistwin_bg, playlistwin_gc, src, 126, 72, w - 150, h - 38, 150, 38);
	}
}

void draw_playlist_window(gboolean force)
{
	gboolean redraw;
	GList *wl;
	Widget *w;

	if (force)
	{
		playlistwin_draw_frame();
		lock_widget_list(playlistwin_wlist);
		draw_widget_list(playlistwin_wlist, &redraw, TRUE);
		
	}
	else
	{
		lock_widget_list(playlistwin_wlist);
		draw_widget_list(playlistwin_wlist, &redraw, FALSE);
	}
	if (redraw || force)
	{
		if (force)
			gdk_window_clear(playlistwin->window);
		else
		{
			wl = playlistwin_wlist;
			while (wl)
			{
				w = (Widget *) wl->data;
				if (w->redraw && w->visible)
				{
					gdk_window_clear_area(playlistwin->window, w->x, w->y, w->width, w->height);
					w->redraw = FALSE;
				}
				wl = wl->next;
			}
		}
		gdk_flush();
	}
	unlock_widget_list(playlistwin_wlist);
}

void playlistwin_sort_menu_callback(gpointer cb_data, guint action, GtkWidget * w)
{
	switch (action)
	{
		case PLAYLISTWIN_SORT_BYTITLE:
			playlist_sort_by_title();
			playlistwin_update_list();
			break;
		case PLAYLISTWIN_SORT_BYFILENAME:
			playlist_sort_by_filename();
			playlistwin_update_list();
			break;
		case PLAYLISTWIN_SORT_BYPATH:
			playlist_sort_by_path();
			playlistwin_update_list();
			break;
		case PLAYLISTWIN_SORT_SEL_BYTITLE:
			playlist_sort_selected_by_title();
			playlistwin_update_list();
			break;
		case PLAYLISTWIN_SORT_SEL_BYFILENAME:
			playlist_sort_selected_by_filename();
			playlistwin_update_list();
			break;
		case PLAYLISTWIN_SORT_SEL_BYPATH:
			playlist_sort_selected_by_path();
			playlistwin_update_list();
			break;
		case PLAYLISTWIN_SORT_REVERSE:
			playlist_reverse();
			playlistwin_update_list();
			break;
		case PLAYLISTWIN_SORT_RANDOMIZE:
			playlist_random();
			playlistwin_update_list();
			break;
	}
}

void playlistwin_sub_menu_callback(gpointer cb_data, guint action, GtkWidget * w)
{
	switch (action)
	{
		case PLAYLISTWIN_REMOVE_DEAD_FILES:
			playlist_remove_dead_files();
			break;
	}
}


void playlistwin_hide_timer(void)
{
	textbox_set_text(playlistwin_time_min, "   ");
	textbox_set_text(playlistwin_time_sec, "  ");
}

void playlistwin_vis_enable(void)
{
	playlistwin_vis_enabled = TRUE;
	if (cfg.playlist_width >= 350)
		show_widget(playlistwin_vis);
}

void playlistwin_vis_disable(void)
{
	playlistwin_vis_enabled = FALSE;
	hide_widget(playlistwin_vis);
	draw_playlist_window(TRUE);
}

void playlistwin_set_time(gint time, gint length, TimerMode mode)
{
	gchar *text, sign;

	if (mode == TIMER_REMAINING && length != -1)
	{
		time = length - time;

		sign = '-';
	}
	else
		sign = ' ';

	time /= 1000;

	if (time < 0)
		time = 0;
	if (time > 99 * 60)
		time /= 60;

	text = g_strdup_printf("%c%-2.2d", sign, time / 60);
	textbox_set_text(playlistwin_time_min, text);
	g_free(text);
	text = g_strdup_printf("%-2.2d", time % 60);
	textbox_set_text(playlistwin_time_sec, text);
	g_free(text);
}

static void playlistwin_drag_data_received(GtkWidget * widget,
					   GdkDragContext * context,
					   gint x,
					   gint y,
					   GtkSelectionData * selection_data,
					   guint info,
					   guint time,
					   gpointer user_data)
{
	guint pos;

	if (selection_data->data)
	{
		if (inside_widget(x, y, playlistwin_list))
		{
			pos = ((y - ((Widget *) playlistwin_list)->y) / playlistwin_list->pl_fheight) + playlistwin_list->pl_first;
			if (pos > get_playlist_length())
				pos = get_playlist_length();
			playlist_ins_url_string((gchar *) selection_data->data, pos);
		}
		else
			playlist_add_url_string((gchar *) selection_data->data);

		playlistwin_update_list();
	}
}

void playlistwin_create(void)
{
	playlistwin = gtk_window_new(GTK_WINDOW_DIALOG);
	dock_add_window(dock_window_list, playlistwin);
	gtk_widget_set_app_paintable(playlistwin, TRUE);
	gtk_window_set_policy(GTK_WINDOW(playlistwin), FALSE, FALSE, TRUE);
	gtk_window_set_title(GTK_WINDOW(playlistwin), _("XMMS Playlist"));
	gtk_window_set_wmclass(GTK_WINDOW(playlistwin), "XMMS_Playlist", "xmms");
	gtk_window_set_transient_for(GTK_WINDOW(playlistwin), GTK_WINDOW(mainwin));
	if (cfg.playlist_x != -1 && cfg.save_window_position)
		dock_set_uposition(playlistwin, cfg.playlist_x, cfg.playlist_y);
	gtk_widget_set_usize(playlistwin, cfg.playlist_width, cfg.playlist_shaded ? 14 : cfg.playlist_height);
	gtk_widget_set_events(playlistwin, GDK_FOCUS_CHANGE_MASK | GDK_BUTTON_MOTION_MASK | GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK);
	gtk_widget_realize(playlistwin);
	hint_set_skip_winlist(playlistwin);

	playlistwin_sort_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", NULL);
#ifdef ENABLE_NLS
{
	int i;
	for (i=0; i < PLAYLISTWIN_SORT_MENU_ENTRIES; i++)
	{
		playlistwin_sort_menu_entries[i].path =
			gettext(playlistwin_sort_menu_entries[i].path);
	}
	playlistwin_sub_menu_entries[0].path =
		gettext(playlistwin_sub_menu_entries[0].path);
}
#endif
	gtk_item_factory_create_items(GTK_ITEM_FACTORY(playlistwin_sort_menu), PLAYLISTWIN_SORT_MENU_ENTRIES, playlistwin_sort_menu_entries, NULL);
	playlistwin_sub_menu = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", NULL);
	gtk_item_factory_create_items(GTK_ITEM_FACTORY(playlistwin_sub_menu), PLAYLISTWIN_SUB_MENU_ENTRIES, playlistwin_sub_menu_entries, NULL);

	gtk_signal_connect(GTK_OBJECT(playlistwin), "delete_event", GTK_SIGNAL_FUNC(playlistwin_delete), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "button_press_event", GTK_SIGNAL_FUNC(playlistwin_press), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "button_release_event", GTK_SIGNAL_FUNC(playlistwin_release), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "motion_notify_event", GTK_SIGNAL_FUNC(playlistwin_motion), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "focus_in_event", GTK_SIGNAL_FUNC(playlistwin_focus_in), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "focus_out_event", GTK_SIGNAL_FUNC(playlistwin_focus_out), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "configure_event", GTK_SIGNAL_FUNC(playlistwin_configure), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "client_event", GTK_SIGNAL_FUNC(playlistwin_client_event), NULL);
	xmms_drag_dest_set(playlistwin);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "drag-data-received", GTK_SIGNAL_FUNC(playlistwin_drag_data_received), NULL);
	gtk_signal_connect(GTK_OBJECT(playlistwin), "key-press-event", GTK_SIGNAL_FUNC(playlistwin_keypress), NULL);

	gdk_window_set_decorations(playlistwin->window, 0);

	playlistwin_bg = gdk_pixmap_new(playlistwin->window, cfg.playlist_width, cfg.playlist_height, gdk_rgb_get_visual()->depth);
	gdk_window_set_back_pixmap(playlistwin->window, playlistwin_bg, 0);
	playlistwin_create_mask();
	playlistwin_gc = gdk_gc_new(playlistwin->window);

	playlistwin_sinfo = create_textbox(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, 4, 4, cfg.playlist_width - 35, FALSE, SKIN_TEXT);
	if (!cfg.playlist_shaded)
		hide_widget(playlistwin_sinfo);
	if (cfg.playlist_shaded)
		playlistwin_shade = create_pbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 21, 3, 9, 9, 128, 45, 150, 42, playlistwin_shade_cb, SKIN_PLEDIT);
	else
		playlistwin_shade = create_pbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 21, 3, 9, 9, 157, 3, 62, 42, playlistwin_shade_cb, SKIN_PLEDIT);
	playlistwin_shade->pb_allow_draw = FALSE;
	playlistwin_close = create_pbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 11, 3, 9, 9, cfg.playlist_shaded ? 138 : 167, cfg.playlist_shaded ? 45 : 3, 52, 42, playlistwin_hide, SKIN_PLEDIT);
	playlistwin_close->pb_allow_draw = FALSE;
	playlistwin_list = create_playlist_list(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, 12, 20, cfg.playlist_width - 31, cfg.playlist_height - 58);
	playlistwin_slider = create_playlistslider(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 15, 20, cfg.playlist_height - 58, playlistwin_list);
	playlistwin_time_min = create_textbox(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 82, cfg.playlist_height - 15, 15, FALSE, SKIN_TEXT);
	playlistwin_time_sec = create_textbox(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 64, cfg.playlist_height - 15, 10, FALSE, SKIN_TEXT);
	playlistwin_info = create_textbox(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 143, cfg.playlist_height - 28, 85, FALSE, SKIN_TEXT);
	playlistwin_vis = create_vis(&playlistwin_wlist, playlistwin_bg, playlistwin->window, playlistwin_gc, cfg.playlist_width - 223, cfg.playlist_height - 26, 72, FALSE);
	hide_widget(playlistwin_vis);

	playlistwin_srew = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 144, cfg.playlist_height - 16, 8, 7, playlist_prev);
	playlistwin_splay = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 138, cfg.playlist_height - 16, 10, 7, mainwin_play_pushed);
	playlistwin_spause = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 128, cfg.playlist_height - 16, 10, 7, input_pause);
	playlistwin_sstop = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 118, cfg.playlist_height - 16, 9, 7, mainwin_stop_pushed);
	playlistwin_sfwd = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 109, cfg.playlist_height - 16, 8, 7, playlist_next);
	playlistwin_seject = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 100, cfg.playlist_height - 16, 9, 7, mainwin_eject_pushed);
	playlistwin_sscroll_up = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 14, cfg.playlist_height - 35, 8, 5, playlistwin_scroll_up_pushed);
	playlistwin_sscroll_down = create_sbutton(&playlistwin_wlist, playlistwin_bg, playlistwin_gc, cfg.playlist_width - 14, cfg.playlist_height - 30, 8, 5, playlistwin_scroll_down_pushed);

	playlistwin_update_info();
}

void playlistwin_show(void)
{	
	gtk_widget_show(playlistwin);
	if (pposition_broken && cfg.playlist_x != -1 && cfg.save_window_position)
		dock_set_uposition(playlistwin, cfg.playlist_x, cfg.playlist_y);
	gtk_widget_set_usize(playlistwin, cfg.playlist_width, cfg.playlist_shaded ? 14 : cfg.playlist_height);
	gdk_flush();
	draw_playlist_window(TRUE);
	tbutton_set_toggled(mainwin_pl, TRUE);
	cfg.playlist_visible = TRUE;
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(mainwin_general_menu, _("/Playlist Editor")))->active = TRUE;
	playlistwin_set_toprow(0);
	playlist_check_pos_current();
	mainwin_set_always_on_top(cfg.always_on_top);
	hint_set_sticky(cfg.sticky);
	hint_set_skip_winlist(playlistwin);
}

void playlistwin_hide(void)
{
	gtk_widget_hide(playlistwin);
	cfg.playlist_visible = FALSE;
	tbutton_set_toggled(mainwin_pl, FALSE);
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(mainwin_general_menu, _("/Playlist Editor")))->active = FALSE;
}
