/* -
 * Copyright (c) 2002, Ramsey G. Brenner <rgbrenner@myrealbox.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Redistributions of products derived from this software must obtain
 *    the specific prior written permission of the author.
 * 4. Neither the name of the author nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

static const char copyright[] =
"@(#) Copyright (c) 2002\n\
  Ramsey G. Brenner <rgbrenner@myrealbox.com>. All rights reserved.\n";

#include "hdrs.h"

/*
 *   There is a single difference between XORFakeKey
 * and XOREncrypt/XORDecrypt: Auto-naming of fake keys
 * use the extension ".key" instead of ".dat".
 *   POSTFIX is defined when this file is compiled
 * to determine which extension is to be used.
 */
/* Encrypt/Decrypt */
int
main(int argc, char **argv)
{
    int    i,
           j,
           size,
           nfiles;
    char  *out_name,
          *tmp_name,
         **in_name;
    FILE **in_fp,
          *out_fp;

    /* Init some variables */
    out_name = NULL;
    prog_name = argv[0];
    size = -1;
    nfiles = 0;
    in_name = Malloc(1);
    in_name[0] = NULL;

    /* Init options */
    for(i = 1; i < argc; i++, optind++) {
      /* Set hash_type */
      if(strcmp(argv[i], "-sha1") == 0) {
        hash_type = SH;
      } else if(strcmp(argv[i], "-md5") == 0) {
        hash_type = MD;

      /* Get output prefix */
      } else if(strncmp(argv[i], "-o", 2) == 0) {
        j = GetOpt(argv + i);
        out_name = optarg;
        i += j;

      /* Set size of output file */
      } else if(strncmp(argv[i], "-s", 2) == 0) {
        j = GetOpt(argv + i);
        if(strisdigit(optarg)) {
          size = atoi(optarg);
        } else {
          usage();
          exit(EXIT_FAILURE);
        }
        i += j;

      /* Add to file list */
      } else {
        nfiles++;
        in_name = Realloc(in_name, (nfiles + 1) * sizeof(char *));
        in_name[nfiles] = NULL;
        in_name[nfiles - 1] = argv[i];
      }
    }

    /* Check usage */
    if(nfiles == 0) {
      usage();
      exit(EXIT_FAILURE);
    }

    /* Init in_fp with file pointers */
    /* Allocate memory for file pointers */
    in_fp = Malloc((nfiles + 1) * sizeof(FILE *));
    in_fp[nfiles] = NULL;
    for(i = 0; in_name[i] != NULL; i++) {
      /* Check if file exists */
      if(!chk_file(in_name[i])) {
        fprintf(stderr, "%s: Could not find file `%s'!\n",
          prog_name, in_name[i]);
        exit(EXIT_FAILURE);
      }

      /* Check if the hash in the filename
         matches its actual hash */
      if(!chk_hash(in_name[i])) {
        fprintf(stderr, "%s: Warning: Hash mismatch for file %s!\n",
          prog_name, in_name[i]);
      }

      /* Open file */
      in_fp[i] = Fopen(in_name[i], "r");
    }

    /* Init out_fp */
    if(out_name == NULL) {
      Asprintf(&tmp_name, "tmpkey.XXXXXX");
      /* Find temp name, and open */
      out_fp = tmp_fp(tmp_name);
    } else {
      /* If -o <name> was specified */
      /* Make sure we aren't about to overwrite one
         of the users other files */
      if(chk_file(out_name)) {
        /* We found a user file named the same as
           our output file */
        /* Ask user if we should overwrite */
        printf("%s: Output file `%s' already exists!\n",
          prog_name, out_name);
        switch(ask_user("Overwrite? [y,N] ")) {
          case 'y':
            break;
          case 'n':
          case '\n':
            exit(EXIT_FAILURE);
        }
      }
      /* Open out filename */
      out_fp = Fopen(out_name, "w");
    }

    /* XOR files */
    XORFiles(in_fp, out_fp, size);

    /* Close all files */
    for(i = 0; in_fp[i] != NULL; i++) {
      fclose(in_fp[i]);
    }
    fclose(out_fp);

    /* Rename output file */
    if(out_name == NULL) {
      if((out_name = hash_rename(tmp_name, POSTFIX)) == NULL) {
        fprintf(stderr, "%s: Warning: Could not rename %s!\n",
          prog_name, tmp_name);
      } else {
        /* Tell user about new file */
        printf("Wrote %s\n", out_name);
      }
    } else {
      /* Don't rename file if user specified name */
      /* Give user hash of new file */
      printf("%s Hash of %s: %s\n",
        get_hash_type(),
        out_name, hash_file(out_name));
    }

    exit(EXIT_SUCCESS);
}

void
usage(void)
{
    fprintf(stderr, "\n"
      "XORCrypt %s\n"
      "Usage: %s [options] <input files>\n"
      "Options:\n"
        "\t-o <name>        - Output to file <name>\n"
        "\t-s <size>        - Output file will be <size> bytes\n"
        "\t-sha1            - Set hash type to SHA-1 (default)\n"
        "\t-md5             - Set hash type to MD5\n"
        "\n",
      VERSION,
      prog_name);

}
