/* -
 * Copyright (c) 2002, Ramsey G. Brenner <rgbrenner@myrealbox.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Redistributions of products derived from this software must obtain
 *    the specific prior written permission of the author.
 * 4. Neither the name of the author nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

static const char copyright[] =
"@(#) Copyright (c) 2002\n\
  Ramsey G. Brenner <rgbrenner@myrealbox.com>. All rights reserved.\n";

#include "hdrs.h"

/* Create keys */
int
main(int argc, char **argv)
{
    int    i,
           j,
           nkeys,
           length;
    char  *prefix;

    /* Init some variables */
    length = -1;
    nkeys = 1;
    prefix = NULL;
    prog_name = argv[0];

    /* Init options */
    for(i = 1; i < argc; i++, optind++) {
      /* Set bgen_type */
      if(strcmp(argv[i], "-ssl") == 0) {
        bgen_type = SSL;
      } else if(strcmp(argv[i], "-urandom") == 0) {
        bgen_type = URANDOM;

      /* Set hash_type */
      } else if(strcmp(argv[i], "-sha1") == 0) {
        hash_type = SH;
      } else if(strcmp(argv[i], "-md5") == 0) {
        hash_type = MD;

      /* Get number of output files */
      } else if(strncmp(argv[i], "-n", 2) == 0) {
        j = GetOpt(argv + i);
        if(strisdigit(optarg)) {
          nkeys = atoi(optarg);
        } else {
          usage();
          exit(EXIT_FAILURE);
        }
        i += j;

      /* Get output prefix */
      } else if(strncmp(argv[i], "-o", 2) == 0) {
        j = GetOpt(argv + i);
        prefix = optarg;
        i += j;

      /* Else, must be length */
      } else {
        if(length == -1) {
          if(strisdigit(argv[i])) {
            length = atoi(argv[i]);
          } else {
            usage();
            exit(EXIT_FAILURE);
          }
        } else {
          usage();
          exit(EXIT_FAILURE);
        }
      }
    }

    /* Check usage */
    if(length == -1) {
      usage();
      exit(EXIT_FAILURE);
    }

    /* Start key generation */
    for(i = 0; i < nkeys; i++) {
      char  *out_name,
            *tmp_name;
      FILE  *key;
      enum {    /* Err enums for genkey() */
        OK,
        GEN_ERR,
        WR_ERR,
        OP_ERR
      };

      /* Init output name and fp */
      if(prefix == NULL) {
        /* Get temporary name */
        Asprintf(&tmp_name, "tmpkey.XXXXXX");
        /* Find temp name, and open */
        key = tmp_fp(tmp_name);
      } else {
        /* Assemble output name */
        Asprintf(&out_name, "%s%d", prefix, i + 1);
        /* Make sure we aren't about to overwrite one
           of the users other files */
        if(chk_file(out_name)) {
          /* We found a file named the same as our output file */
          /* Ask user if we should overwrite */
          printf("%s: Output file `%s' already exists!\n",
            prog_name, out_name);
          switch(ask_user("Overwrite? [y,N] ")) {
            case 'y':
              break;
            case 'n':
            case '\n':
              printf("Skipping key number `%d'.\n",
                i + 1);
              continue;
          }
        }
        /* Open output file */
        key = Fopen(out_name, "w");
      }

      /* Generate key */
      switch(genkey(key, length)) {
        case GEN_ERR:
          fprintf(stderr, "%s: Error generating byte for %s!\n",
            prog_name, tmp_name);
          exit(EXIT_FAILURE);
        case WR_ERR:
          fprintf(stderr, "%s: Could not write byte to %s!\n",
            prog_name, tmp_name);
          exit(EXIT_FAILURE);
        case OP_ERR:
          perror(prog_name);
          exit(EXIT_FAILURE);
      }

      /* Close key */
      fclose(key);

      /* Rename key */
      if(prefix == NULL) {
        if((out_name = hash_rename(tmp_name, "key")) == NULL) {
          fprintf(stderr, "%s: Warning: Could not rename %s!\n",
            prog_name, tmp_name);
        } else {
          /* Tell user about new file */
          printf("Wrote %s\n", out_name);
        }

        /* Release tmp_name */
        Free(tmp_name);
      } else {
        /* Don't rename file if user specified prefix */
        /* Give user hash of new file */
        printf("%s Hash of %s: %s\n",
          get_hash_type(),
          out_name, hash_file(out_name));
      }

      /* Cleanup */
      Free(out_name);
    }
    exit(EXIT_SUCCESS);
}

void
usage(void)
{
    fprintf(stderr, "\n"
      "XORCrypt %s\n"
      "Usage: %s [options] <length>\n"
      "Options:\n"
        "\t-n <number>      - Number of keys to generate (default: 1)\n"
        "\t-o <prefix>      - Set name prefix\n"
        "\t-sha1            - Set hash type to SHA-1 (default)\n"
        "\t-md5             - Set hash type to MD5\n"
        "\t-ssl             - Set byte generator to use SSL (default)\n"
        "\t-urandom         - Set byte generator to use /dev/urandom\n"
        "\n",
      VERSION,
      prog_name);
}
