/* -
 * Copyright (c) 2002, Ramsey G. Brenner <rgbrenner@myrealbox.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Redistributions of products derived from this software must obtain
 *    the specific prior written permission of the author.
 * 4. Neither the name of the author nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

extern char *prog_name;

enum {
    MD,
    SH
} hash_type = SH;

/* Return the hash type in a string */
char *
get_hash_type(void)
{
    switch(hash_type) {
      case MD:
        return("MD5");
      case SH:
      default:
        return("SHA-1");
    }
}

/* Hash file `fn' and return its hash */
char *
hash_file(char *fn)
{
    int             i;
    char            byte;
    FILE           *fp;
    MD5_CTX         md_c;
    SHA_CTX         sh_c;
    static char     sh_hash[SHA_DIGEST_LENGTH * 2];
    static char     md_hash[MD5_DIGEST_LENGTH * 2];
    unsigned char   sh_md[SHA_DIGEST_LENGTH];
    unsigned char   md_md[MD5_DIGEST_LENGTH];

    /* open file */
    fp = Fopen(fn, "r");

    switch(hash_type) {
      case MD:  /* MD5 */
        MD5_Init(&md_c);
        for(;;) {
          if(!fread(&byte, sizeof(byte), 1, fp)) {
            if(feof(fp)) {
              fclose(fp);
              break;
            } else {
              fprintf(stderr, "%s: Error reading byte!\n",
                prog_name);
              exit(EXIT_FAILURE);
            }
          }
          MD5_Update(&md_c, &byte, 1);
        }
        MD5_Final(md_md, &md_c);
        for(i = 0; i < MD5_DIGEST_LENGTH; i++) {
          sprintf(md_hash + i + i, "%02x", md_md[i]);
        }
        return(md_hash);
      case SH: /* SHA-1 */
      default:
        SHA1_Init(&sh_c);
        for(;;) {
          if(!fread(&byte, sizeof(byte), 1, fp)) {
            if(feof(fp)) {
              fclose(fp);
              break;
            } else {
              fprintf(stderr, "%s: Error reading byte!\n",
                prog_name);
              exit(EXIT_FAILURE);
            }
          }
          SHA1_Update(&sh_c, &byte, 1);
        }
        SHA1_Final(sh_md, &sh_c);
        for(i = 0; i < SHA_DIGEST_LENGTH; i++) {
          sprintf(sh_hash + i + i, "%02x", sh_md[i]);
        }
        return(sh_hash);
    }
}

/* Check if the hash in the filename
   equals its real hash */
int
chk_hash(char *fn)
{
    int      i;
    int      htype_bckup;
    char    *hash;
    char    *fhash;
    char     fn_cpy[strlen(fn) + 1];
    char    *filename;
    enum {
      MISMATCH,
      MATCH
    };

    /* Init htype_bckup */
    htype_bckup = hash_type;

    /* Copy fn tp fn_cpy */
    strcpy(fn_cpy, fn);

    /* Extract filename from fn_cpy */
    filename = fn_cpy;
    for(i = strlen(fn_cpy); i >= 0; i--) {
      if(fn_cpy[i] == '/') {
        fn_cpy[i] = NULL;
        filename = &fn_cpy[i + 1];
        break;
      }
    }

    /* Set hash type and extract hash from filename */
    if(strncmp(filename, "sha1-", strlen("sha1-")) == 0) {
      hash_type = SH;
      fhash = filename + strlen("sha1-");
    } else if(strncmp(filename, "md5-", strlen("md5-")) == 0) {
      hash_type = MD;
      fhash = filename + strlen("md5-");
    } else {
      /* Assume user renamed file */
      /* Restore hash_type */
      hash_type = htype_bckup;
      return(MATCH);
    }

    /* Remove postfix */
    for(i = 0; fhash[i] != NULL; i++) {
      if(fhash[i] == '.') {
        fhash[i] = NULL;
        break;
      }
    }

    /* Hash file */
    hash = hash_file(fn);

    /* Restore hash_type */
    hash_type = htype_bckup;

    /* Compare hashes */
    if(strcmp(hash, fhash) == 0 &&
        strlen(hash) == strlen(fhash)) {
      return(MATCH);
    }
    return(MISMATCH);
}

/* Rename file with the format
   <hash type>-<hash>.<postfix> */
char *
hash_rename(char *fn, char *postfix)
{
    char  *hash_str;
    char  *new_name;

    /* Set hash_str */
    switch(hash_type) {
      case MD:
        hash_str = "md5";
        break;
      case SH:
      default:
        hash_str = "sha1";
        break;
    }

    /* Create new file name */
    Asprintf(&new_name, "%s-%s.%s",
        hash_str, hash_file(fn), postfix);

    /* Rename file */
    if(rename(fn, new_name) < 0) {
      Free(new_name);
      return(NULL);
    }

    return(new_name);
}
