/* -
 * Copyright (c) 2002, Ramsey G. Brenner <rgbrenner@myrealbox.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Redistributions of products derived from this software must obtain
 *    the specific prior written permission of the author.
 * 4. Neither the name of the author nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

static const char copyright[] =
"@(#) Copyright (c) 2002\n\
  Ramsey G. Brenner <rgbrenner@myrealbox.com>. All rights reserved.\n";

#include "hdrs.h"

void   split(FILE *, FILE **, FILE *, int, int, int);

/* Split or combine file(s) */
int
main(int argc, char **argv)
{
    int     i,
            j,
            size,
            nkeys,
            nfiles;
    char  **files,
           *output_filename;

    /* Init some variables */
    nkeys = 1;
    nfiles = 0;
    size = -1;
    output_filename = NULL;
    prog_name = argv[0];
    files = Malloc(sizeof(char *));
    files[0] = NULL;

    /* Init options */
    for(i = 1; i < argc; i++, optind++) {
      /* Set bgen_type */
      if(strcmp(argv[i], "-ssl") == 0) {
        bgen_type = SSL;
      } else if(strcmp(argv[i], "-urandom") == 0) {
        bgen_type = URANDOM;

      /* Set hash_type */
      } else if(strcmp(argv[i], "-sha1") == 0) {
        hash_type = SH;
      } else if(strcmp(argv[i], "-md5") == 0) {
        hash_type = MD;

      /* Get number of output files */
      } else if(strncmp(argv[i], "-n", 2) == 0) {
        j = GetOpt(argv + i);
        if(strisdigit(optarg)) {
          nkeys = atoi(optarg);
        } else {
          usage();
          exit(EXIT_FAILURE);
        }
        i += j;

      /* Set size of output file */
      } else if(strncmp(argv[i], "-s", 2) == 0) {
        j = GetOpt(argv + i);
        if(strisdigit(optarg)) {
          size = atoi(optarg);
        } else {
          usage();
          exit(EXIT_FAILURE);
        }
        i += j;

      /* Get output filename (only used if user is combining files) */
      } else if(strncmp(argv[i], "-o", 2) == 0) {
        j = GetOpt(argv + i);
        output_filename = optarg;
        i += j;

      /* Add to file list */
      } else {
        nfiles++;
        files = Realloc(files, (nfiles + 1) * sizeof(char *));
        files[nfiles] = NULL;
        files[nfiles - 1] = argv[i];
      }
    }
    argc -= optind;

    /* Check usage */
    if(nfiles == 0) {
      usage();
      exit(EXIT_FAILURE);
    }

    /* split/combine file(s) */
    if(nfiles == 1) {  /* Split file */
      char   *in_name,
             *out_name,
            **key_name;
      FILE   *in_fp,
             *out_fp,
            **key_fp;

      /* Assign name to in_name */
      in_name = files[0];
      /* Make sure input file exists */
      if(!chk_file(in_name)) {
        fprintf(stderr, "%s: Could not find file `%s'!\n",
          prog_name, in_name);
        exit(EXIT_FAILURE);
      }
      /* Open input file */
      in_fp = Fopen(in_name, "r");

      /* Allocate space to store key file names */
      key_name = Malloc((nkeys + 1) * sizeof(char *));
      /* Allocate space to store key FILE ptrs */
      key_fp = Malloc((nkeys + 1) * sizeof(FILE *));
      /* Assign NULLs to end of space */
      key_name[nkeys] = NULL;
      key_fp[nkeys] = NULL;
      /* Open key files */
      for(i = 0; i < nkeys; i++) {
        /* Allocate space for temp name */
        Asprintf(&key_name[i], "tmpkey.XXXXXX");
        /* Find temp name, and open */
        key_fp[i] = tmp_fp(key_name[i]);
      }

      /* Open output file */
      /* Allocate space for temp name */
      Asprintf(&out_name, "tmpkey.XXXXXX");
      /* Find temp name, and open */
      out_fp = tmp_fp(out_name);

      /* Split file */
      split(in_fp, key_fp, out_fp, get_length(in_name), nkeys, size);

      /* Close outputed files */
      /* Close input file */
      fclose(in_fp);
      /* Close key files */
      for(i = 0; i < nkeys; i++) {
        fclose(key_fp[i]);
      }
      /* Close output file */
      fclose(out_fp);

      /* Cleanup space allocated for FILE ptrs */
      Free(key_fp);

      /* Rename key files */
      for(i = 0; i < nkeys; i++) {
        char  *name;

        /* Rename key */
        if((name = hash_rename(key_name[i], "key")) == NULL) {
          fprintf(stderr, "%s: Warning: Could not rename %s!\n",
            prog_name, key_name[i]);
        } else {
          /* Tell user about new file */
          printf("Wrote %s\n", name);
          Free(name);
        }
      }

      /* Rename output file */
      {
        char  *name;

        if((name = hash_rename(out_name, "dat")) == NULL) {
          fprintf(stderr, "%s: Warning: Could not rename %s!\n",
            prog_name, out_name);
        } else {
          /* Tell user about new file */
          printf("Wrote %s\n", name);
          Free(name);
        }
      }
    } else {  /* Combine files */
      int     n_infp; /* Number of input files */
      char  **in_name,
             *out_name = output_filename;
      FILE  **in_fp,
             *out_fp;

      /* Init n_infp */
      n_infp = nfiles;

      /* Allocate space for input names */
      in_name = Malloc((n_infp + 1) * sizeof(char *));
      /* Allocate space to store FILE ptrs */
      in_fp = Malloc((n_infp + 1) * sizeof(FILE *));
      /* Assign NULL to end of memory */
      in_name[n_infp] = NULL;
      in_fp[n_infp] = NULL;

      /* Assign names from files[] to in_name */
      for(i = 0; i < n_infp; i++) {
        in_name[i] = files[i];
      }

      for(i = 0; i < n_infp; i++) {
        /* Make sure all the files exist */
        if(!chk_file(in_name[i])) {
          fprintf(stderr, "%s: Could not find file `%s'!\n",
            prog_name, in_name[i]);
          exit(EXIT_FAILURE);
        }

        /* Check filename hashes */
        if(!chk_hash(in_name[i])) {
          fprintf(stderr, "%s: Warning: Hash mismatch for file %s!\n",
            prog_name, in_name[i]);
        }

        /* Open input files */
        in_fp[i] = Fopen(in_name[i], "r");
      }

      /* Init out_fp */
      if(output_filename == NULL) {
        /* Print to stdout */
        out_fp = stdout;
      } else {
        /* If -o <name> was specified */
        /* Make sure we aren't about to overwrite one
           of the users other files */
        if(chk_file(out_name)) {
          /* We found a user file named the same as
             our output file */
          /* Ask user if we should overwrite */
          printf("%s: Output file `%s' already exists!\n",
            prog_name, out_name);
          switch(ask_user("Overwrite? [y,N] ")) {
            case 'y':
              break;
            case 'n':
            case '\n':
              exit(EXIT_FAILURE);
          }
        }
        /* Open out filename */
        out_fp = Fopen(out_name, "w");
      }

      /* XOR files */
      XORFiles(in_fp, out_fp, size);

      /* Close all files */
      for(i = 0; in_fp[i] != NULL; i++) {
        fclose(in_fp[i]);
      }
      fclose(out_fp);

      /* Print hash */
      if(output_filename != NULL) {
        /* -o <name> was specified */
        /* Print hash */
        printf("%s Hash of %s: %s\n",
          get_hash_type(),
          out_name, hash_file(out_name));
      }
    }

    exit(EXIT_SUCCESS);
}

void
split(FILE *in_fp, FILE **key_fp, FILE *out_fp,
      int length, int nkeys, int size)
{
    int     i,
            nfp;  /* number of input files */
    FILE  **input_fp;
    enum {    /* Err enums for genkey() */
      OK,
      GEN_ERR,
      WR_ERR,
      OP_ERR
    };

    /* Generate keys */
    for(i = 0; i < nkeys; i++) {
      switch(genkey(key_fp[i], size != -1 ? size : length)) {
        case GEN_ERR:
          fprintf(stderr, "%s: Error generating byte!\n",
            prog_name);
          exit(EXIT_FAILURE);
        case WR_ERR:
          fprintf(stderr, "%s: Could not write byte!\n",
            prog_name);
          exit(EXIT_FAILURE);
        case OP_ERR:
          perror(prog_name);
          exit(EXIT_FAILURE);
      }
    }

    /* Rewind key files */
    for(i = 0; i < nkeys; i++) {
      rewind(key_fp[i]);
    }

    /* nfp = nkeys + 1 for in_fp */
    nfp = nkeys + 1;

    /* Allocate enough memory to hold all the input FILE ptrs */
    /* nkeys + 1 for in_fp + 1 for NULL */
    input_fp = Malloc((nfp + 1) * sizeof(FILE *));
    /* Assign NULL to end of memory */
    input_fp[nfp] = NULL;

    /* group in_fp and key_fp */
    input_fp[0] = in_fp;
    for(i = 1; i < nfp; i++) {
      input_fp[i] = key_fp[i - 1];
    }

    /* XOR files */
    XORFiles(input_fp, out_fp, size);

    /* cleanup and return */
    Free(input_fp);
    return;
}

void
usage(void)
{
    fprintf(stderr, "\n"
      "XORCrypt %s\n"
      "Usage: %s [options] <input files>\n"
      "Options:\n"
        "\t-n <number>      - Number of keys to mix in (default: 1)\n"
        "\t-o <name>        - Output name of combined files (default: stdout)\n"
        "\t-s <size>        - Output file will be <size> bytes\n"
        "\t-sha1            - Set hash type to SHA-1 (default)\n"
        "\t-md5             - Set hash type to MD5\n"
        "\t-ssl             - Set byte generator to use SSL (default)\n"
        "\t-urandom         - Set byte generator to use /dev/urandom\n"
        "\n",
      VERSION,
      prog_name);
}
