/* -
 * Copyright (c) 2002, Ramsey G. Brenner <rgbrenner@myrealbox.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Redistributions of products derived from this software must obtain
 *    the specific prior written permission of the author.
 * 4. Neither the name of the author nor the names of its contributors may
 *    be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

extern char *prog_name;

/* Create a temporary file and return a FILE ptr */
FILE *
tmp_fp(char *name)
{
    int    fd;
    FILE  *fp;

    if((fd = mkstemp(name)) < 0) {
      perror(prog_name);
      exit(EXIT_FAILURE);
    }
    if((fp = fdopen(fd, "r+")) == NULL) {
      close(fd);
      remove(name);
      perror(prog_name);
      exit(EXIT_FAILURE);
    }
    return(fp);
}

/* Check if file exists */
int
chk_file(char *fn)
{
    int              i;
    char            *path,
                    *filename;
    DIR             *dp;
    struct dirent   *dirp;
    enum {
      FALSE,
      TRUE
    } found = FALSE;

    /* Seperate path and filename */
    path = ".";
    filename = fn;
    for(i = strlen(fn); i > 0; i--) {
      if(fn[i - 1] == '/') {
        path = fn;
        filename = &fn[i];
        fn[i - 1] = NULL;
        break;
      }
    }

    /* Open directory */
    if((dp = opendir(path)) == NULL) {
        fprintf(stderr, "%s: Error opening `%s'!\n",
          prog_name, path, filename);
        exit(EXIT_FAILURE);
    }

    /* Find file */
    while((dirp = readdir(dp)) != NULL) {
        if(strcmp(dirp->d_name, filename) == 0) {
            found = TRUE;
            break;
        }
    }

    /* Cleanup and return */
    closedir(dp);
    return(found);
}

/* Check if entire string is a digit */
int
strisdigit(char *s)
{
    int i;

    for(i = 0; i < s[i] != NULL; i++) {
      if(!isdigit(s[i])) {
        return(0);
      }
    }
    return(1);
}

/* Ask user yes/no question */
int
ask_user(char *msg)
{
    for(;;) {
      int  c;

      /* Print question */
      printf("%s", msg);
      /* Clear stdin before we read the users answer */
      fpurge(stdin);
      if((c = getchar()) == EOF) {
        perror(prog_name);
        exit(EXIT_FAILURE);
      }
      /* Clear stdin again to erase any extra characters
         we don't need */
      fpurge(stdin);
      switch(c) {
        case 'Y':
        case 'N':
          c = tolower(c);
        case 'y':
        case 'n':
        case '\n': /* User selected default answer */
          return(c);
      }
      /* User didn't answer correctly, so we'll ask again */
      continue;
    }
}

/* Return filename length */
int
get_length(char *filename)
{
    struct stat sb;

    if(stat(filename, &sb) < 0) {
      perror(prog_name);
      exit(EXIT_FAILURE);
    }
    return(sb.st_size);
}

/* Our getopt() */
int    optind = 0;
char  *optarg = NULL;
int
GetOpt(char **arg)
{
    if(strlen(arg[0]) > 2) {
      optarg = arg[0] + 2;
      return(0);
    } else {
      optarg = arg[1];
      optind++;
      return(1);
    }
}

/* fopen() */
FILE *
Fopen(const char *path, const char *mode)
{
    FILE  *tmp;

    if((tmp = fopen(path, mode)) == NULL) {
      perror(prog_name);
      exit(EXIT_FAILURE);
    }
    return(tmp);
}

/* atexit() */
void
AtExit(void (*function)(void))
{
    if(atexit(function) < 0) {
      perror(prog_name);
      exit(EXIT_FAILURE);
    }
}

/* Our asprintf() */
void
Asprintf(char **ret, char *fmt, ...)
{
    int      d,
             nbytes;
    char     c,
            *s,
            *ptr,
            *str;
    va_list  ap;

    /* Init */
    va_start(ap, fmt);
    nbytes = 1;
    str = Malloc(1);
    *str = NULL;

    /* Parse fmt */
    for(; *fmt != NULL; fmt++) {
      if(*fmt == '%') {
        fmt++;
        switch(*fmt) {
          case 'd':
            d = va_arg(ap, int);
            ptr = Malloc(nbytes + sizeof(int));
            sprintf(ptr, "%s%d", str, d);
            nbytes++;
            break;
          case 'c':
            c = va_arg(ap, int);
            ptr = Malloc(nbytes + 1);
            sprintf(ptr, "%s%c", str, c);
            nbytes++;
            break;
          case 's':
            s = va_arg(ap, char *);
            ptr = Malloc(nbytes + strlen(s));
            sprintf(ptr, "%s%s", str, s);
            nbytes += strlen(s);
            break;
        }
      } else {
        ptr = Malloc(nbytes + 1);
        sprintf(ptr, "%s%c", str, *fmt);
        nbytes++;
      }
      Free(str);
      str = ptr;
    }

    /* Init ret */
    *ret = str;

    /* Cleanup and return */
    va_end(ap);
    return;
}
