
/* +-------------------------------------------------------------------+ */
/* | Copyright 2000, J.-P. Demailly (demailly@fourier.ujf-grenoble.fr) | */
/* |                                                                   | */
/* | Permission to use, copy, modify, and to distribute this software  | */
/* | and its documentation for any purpose is hereby granted without   | */
/* | fee, provided that the above copyright notice appear in all       | */
/* | copies and that both that copyright notice and this permission    | */
/* | notice appear in supporting documentation.  There is no           | */
/* | representations about the suitability of this software for        | */
/* | any purpose.  this software is provided "as is" without express   | */
/* | or implied warranty.                                              | */
/* |                                                                   | */
/* +-------------------------------------------------------------------+ */

/* $Id: splineOp.c,v 1.2 2000/10/02 21:06:44 torsten Exp $ */

#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#include "xpaint.h"
#include "misc.h"
#include "Paint.h"
#include "ops.h"

#define MAXP	200
#define SUBDIV	50
#define FLOAT float

enum {OPEN=0, CLOSED, CLOSEDUP};
enum {FINISH, DOALL, ERASE, DRAW};

static int Mode = 0;

typedef struct {
    int mode, npoints, nprev;
    XPoint bottom, top;
    XPoint points[MAXP + 2];
    XPoint interm1[MAXP + 2];
    XPoint interm2[MAXP + 2];
    /*
    **  Borrowed from my info structure.
     */
    GC gcx, lgc;
    Pixmap pixmap;
    Widget widget;
    Boolean isFat;
} LocalInfo;

void
SplineSetMode(int mode)
{
    Mode = mode;
}

static void 
UpdateSave(LocalInfo *l, short u, short v)
{
    if (u < l->bottom.x) l->bottom.x = u;
    if (v < l->bottom.y) l->bottom.y = v;
    if (u > l->top.x) l->top.x = u;
    if (v > l->top.y) l->top.y = v;
}

static void 
XDrawContour(Display *dpy, Window win, GC gc, LocalInfo *l, int flag)
{
    int n, i,j, imax, i0, i1;
    short u, v, up, vp;
    FLOAT ax, ay, bx, by, cx, cy, r, s, t;

    if (flag == ERASE) {
        n = l->nprev;
    }
    else {
        n = l->npoints;
        l->mode = Mode;
    }

    if (n <= 0)
        return;

    if (n == 1) {
         XDrawLine(dpy, win, gc, l->points[0].x, l->points[0].y,
                                 l->points[0].x, l->points[0].y);
         XDrawLine(dpy, win, gc, l->points[0].x, l->points[0].y,
                                 l->points[1].x, l->points[1].y);
         if (flag == FINISH)
             for (i=0; i<=1; i++)
	         UpdateSave(l, l->points[i].x, l->points[i].y);
         l->mode = Mode;
         l->nprev = l->npoints;
         return;
    }

    if (l->mode != OPEN) {
        j = n+2;
        l->points[n+1].x = l->points[0].x;
        l->points[n+1].y = l->points[0].y;
        l->points[n+2].x = l->points[1].x;
        l->points[n+2].y = l->points[1].y;
    }
    else 
        j = n;

    i0 = n-2;
    if (i0 < 0)
        i0 = 0;
    for (i = i0; i < j; i++) {
        s = r;
	bx = ax;
	by = ay;
	ax = l->points[i+1].x - l->points[i].x;
	ay = l->points[i+1].y - l->points[i].y;
	r = ax*ax + ay*ay + 1E-10;
	ax = ax/r; ay = ay/r;
	r = sqrt(r)/3.0;
	if (i > i0) { 
	    cx = (ax+bx)/2; cy = (ay+by)/2; 
	    t = sqrt(cx*cx+cy*cy) + 1E-5;
	    cx = cx/t; cy = cy/t;
	    l->interm1[i].x = l->points[i].x + (short)(r*cx);
	    l->interm1[i].y = l->points[i].y + (short)(r*cy);
	    l->interm2[i-1].x = l->points[i].x - (short)(s*cx);
	    l->interm2[i-1].y = l->points[i].y - (short)(s*cy);
	    if ((i == 1) && (l->mode == OPEN)) {
	        t = 3*s*s;
		l->interm1[0].x = l->interm2[0].x - (short)(t*bx);
		l->interm1[0].y = l->interm2[0].y - (short)(t*by);
	    }
	    if ((i == n-1) && (l->mode == OPEN)) {
	        t = 3*r*r;
		l->interm2[i].x = l->interm1[i].x + (short)(t*ax);
		l->interm2[i].y = l->interm1[i].y + (short)(t*ay);
	    }
	    if (i == n+1) {
	        l->interm1[0].x = l->interm1[i].x;
		l->interm1[0].y = l->interm1[i].y;
	    }
	}
    }
 
    imax = n;
    if (((l->mode == CLOSED) && (flag <= DOALL)) || (l->mode == CLOSEDUP))
        ++imax;

    if (flag <= DOALL) {
        i0 = -1;
	i1 = 0;
    }
    if (flag == ERASE) {
        if (Mode == OPEN)
	    i0 = -1;
	else
            i0 = 0;
	i1 = l->npoints-2;
    }
    if (flag == DRAW) {
        if (Mode == OPEN)
            i0 = -1;
	else
            i0 = 0;
	i1 = n-2;
    } 
    
    for (i = 0; i < imax; i++) {
        if ((i <= i0) || (i >= i1)) {
	    u = l->points[i].x;
	    v = l->points[i].y;
	    for (j = 1; j <= SUBDIV; j++) {
	        up = u;
		vp = v;
		t = ((FLOAT) j)/((FLOAT) SUBDIV);
		s = 1.0-t;
		u = (short) (s*s*(s*l->points[i].x+3.0*t*l->interm1[i].x)+
			     t*t*(t*l->points[i+1].x+3.0*s*l->interm2[i].x));
		v = (short) (s*s*(s*l->points[i].y+3.0*t*l->interm1[i].y)+
		       t*t*(t*l->points[i+1].y+3.0*s*l->interm2[i].y));
		if (flag == FINISH)
		    UpdateSave(l, u, v);
		XDrawLine(dpy, win, gc, up, vp, up, vp);
		XDrawLine(dpy, win, gc, up, vp, u, v);
	    }
	}
    }

    l->mode = Mode;
    l->nprev = l->npoints;
}

static void 
finish(Widget w, LocalInfo * l)
{
    XRectangle rect;

    if (l->npoints < 0)
        return;

    XDrawContour(XtDisplay(w), XtWindow(w), l->gcx, l, ERASE);
    --l->npoints;

    if (!l->isFat && (l->npoints > 0)) {
        XDrawContour(XtDisplay(w), XtWindow(w), l->lgc, l, DOALL);
        XDrawContour(XtDisplay(w), l->pixmap, l->lgc, l, FINISH);
    }
 
    --l->bottom.x; --l->bottom.y;
    ++l->top.x; ++l->top.y;
    rect.x = l->bottom.x;
    rect.y = l->bottom.y;
    rect.width = l->top.x - l->bottom.x;
    rect.height = l->top.y - l->bottom.y;
    UndoGrow(w, l->bottom.x, l->bottom.y);
    UndoGrow(w, l->top.x, l->top.y);
    PwUpdate(w, &rect, False);

    l->npoints = -1;
    l->nprev = -1;
}

static void 
check_shift(XButtonEvent * event)
{
    if (event->state & ControlMask)
         if (Mode == CLOSED) Mode = CLOSEDUP;
    if (!(event->state & ControlMask))
         if (Mode == CLOSEDUP) Mode = CLOSED;
}

static void 
press(Widget w, LocalInfo * l, XButtonEvent * event, OpInfo * info)
{

    if (event->button == Button3) return;
    check_shift(event);

    if ((l->npoints < 0) && (event->button == Button1) &&
	(info->surface == opPixmap)) {

	l->isFat = info->isFat;
	l->lgc = info->first_gc;

	l->npoints = 0;
        l->mode = Mode;
	l->bottom.x = l->top.x = l->points[0].x = event->x;
	l->bottom.y = l->top.y = l->points[0].y = event->y;
        UndoStartPoint(w, info, l->bottom.x, l->bottom.y);
        l->pixmap = info->drawable;
	l->widget = w;
        return;
    }
 
    if ((l->npoints >= 0) && (event->button == Button2)) {
  	finish(w, l);
	return;
    }
}

static void 
release(Widget w, LocalInfo * l, XButtonEvent * event, OpInfo * info)
{

    if (event->button == Button3) return;

    if (l->npoints < 0) 
        return;

    if (l->npoints == MAXP - 1)
	return;

    if ((event->button == Button1) && (info->surface == opWindow)) {
        check_shift(event);
        ++l->npoints;
	l->points[l->npoints].x = event->x;
	l->points[l->npoints].y = event->y;
    }
}

static void 
motion(Widget w, LocalInfo * l, XMotionEvent * event, OpInfo * info)
{
    /*
    **  Haven't done the first button press
     */
    if (l->npoints < 0)
	return;

    if ((event->x != l->points[l->npoints].x) ||
	(event->y != l->points[l->npoints].y)) {
        check_shift((XButtonEvent *)event);
        XDrawContour(XtDisplay(w), info->drawable, l->gcx, l, ERASE);
        l->points[l->npoints].x = event->x;
        l->points[l->npoints].y = event->y;
	XDrawContour(XtDisplay(w), info->drawable, l->gcx, l, DRAW);
    }
}

void *
SplineAdd(Widget w)
{
    LocalInfo *l = (LocalInfo *) XtMalloc(sizeof(LocalInfo));

    l->npoints = -1;
    l->nprev = -1;
    l->gcx = GetGCX(w);

    XtVaSetValues(w, XtNcompress, True, NULL);

    OpAddEventHandler(w, opWindow | opPixmap, ButtonPressMask, FALSE,
		      (OpEventProc) press, l);
    OpAddEventHandler(w, opWindow, PointerMotionMask, FALSE,
		      (OpEventProc) motion, l);
    OpAddEventHandler(w, opWindow | opPixmap, ButtonReleaseMask, FALSE,
		      (OpEventProc) release, l);
    SetCrossHairCursor(w);

    return l;
}

void 
SplineRemove(Widget w, void *l)
{
    OpRemoveEventHandler(w, opWindow | opPixmap, ButtonPressMask, FALSE,
			 (OpEventProc) press, l);
    OpRemoveEventHandler(w, opWindow, PointerMotionMask, FALSE,
			 (OpEventProc) motion, l);
    OpRemoveEventHandler(w, opWindow | opPixmap, ButtonReleaseMask, FALSE,
			 (OpEventProc) release, l);

    finish(w, (LocalInfo *) l);
    XtFree((XtPointer) l);
}
