/*
 * Copyright (c) 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <internal.h>
#include <err.h>

RCSID("$Id: ZRlmCfg.c,v 1.4 2000/08/14 12:46:20 lha Exp $");

/*
 * Fetch next word that is delimited by "\t\n " or 
 * (return NULL and set *s to NULL)
 */

static char *
next_word (char **s)
{
    char *str;

    if (s == NULL) return NULL;

    while (isspace(**s)) (*s)++;
    str = *s;
    if (*str == '\0') {
	*s = NULL;
	return NULL;
    }    
    *s = strpbrk (*s, "\t\n ");
    if (*s == NULL) {
	return str;
    }
    **s = '\0';
    (*s)++;
    return str;
}

static void
new_server (Z_RealmConfig *rc, const char *name)
{
    Z_HostNameAddr *tmp;
    struct hostent *hp;
    
    tmp = realloc(rc->server_list, sizeof(rc->server_list[0]) * (rc->nservers+1));
    if (tmp == NULL)
	errx (1, "new_server: realloc");
    rc->server_list = tmp;
    tmp = &rc->server_list[rc->nservers];
    
    tmp->name = strdup (name);
    if (tmp->name == NULL)
	err (1, "new_server: strdup");

    tmp->flags = 0;
    
    hp = gethostbyname(tmp->name);
    if (hp) {
	/* XXX multihoming ... */
	tmp->flags |= ZHOST_VALID_ADDRESS;
	memcpy (&tmp->addr, hp->h_addr_list[0], sizeof(tmp->addr));
    } else {
	memset (&tmp->addr, 0, sizeof(tmp->addr));
    }
    rc->nservers++;
}
    

/*
 * Give the real configuration line `str', fill in the realm
 * configuration `rc'.
 *
 * Syntax is (in regexp):
 *    ^*#.*$
 *    ^*REALM.NAME +[hostlist +server1 +server2 ... |hesiod clustername|dns]$
 */

Code_t
Z_ParseRealmConfig(const char *config_line, Z_RealmConfig *rc)
{
    char *s, *str, *str_orig;
    char *realm_name, *config_type;

    str_orig = str = s = strdup (config_line);
    if (str == NULL)
	return ENOMEM;

    rc->nservers = 0;
    rc->server_list = NULL;
    rc->realm = NULL;

    while (isspace(*s)) s++;
    
    if (*s == '#' && *s == '\0') {
	free (str_orig);
	return ZERR_BADCONFREALM;
    }
    
    realm_name = next_word (&s);
    if (realm_name == NULL) {
	free (str_orig);
	return ZERR_BADCONFREALM;
    }

    rc->realm = strdup (realm_name);
    if (rc->realm == NULL) {
	free (str_orig);
	return ZERR_BADCONFREALM;
    }

    config_type = next_word (&s);
    if (config_type == NULL)
	config_type = "dns";

    if (strcasecmp ("hostlist", config_type) == 0) {
	while (s) {
	    char *server = next_word (&s);
	    if (server)
		new_server (rc, server);
	}
#if 0
    } else if (strcasecmp ("dns", config_type) == 0) {
	/* check for:
	 *   SRV-rr _zephyr._udp.REALM
	 *   TXT zephyr.REALM
	 *   zephyr.REALM
	 *   z{,ephyr}{1,2,3}.REALM
	 */
	abort();
#endif
#if HAVE_HESIOD
    } else if (strcasecmp ("hesiod", config_type) == 0) {
	char **hesiod_servers;
	char *clustername;

	clustername = next_word (&s);
	if (clustername == NULL) {
	    syslog (LOG_ERR, "No hesiod cluster for realm %s", rc->realm);
	    free (rc->realm);
	    free (str_orig);
	    return ZERR_BADCONFREALM;
	}
	
	hesiod_servers = hes_resolve(clustername, "sloc");
	if (hesiod_servers == NULL) {
	    syslog (LOG_ERR, "No hesiod server for realm %s (%s sloc)",
		    rc->realm, clustername);
	    free (clustername);
	    free (rc->realm);
	    free (str_orig);
	    return ZERR_BADCONFREALM;
	}
	while (*hesiod_servers) {
	    new_server (rc, *hesiod_servers);
	    hesiod_servers++;
	}
#endif
    } else {
	warnx ("unknown type %s for realm %s", config_type, realm_name);
	/* XXX unknown */
    }
    free (str_orig);

#ifdef HAVE_KRB4
    /*
     * If default-realm is used, go fetch it from one of the servers
     */
    if (strcmp (rc->realm, Z_DEFAULT_REALM) == 0) {
	int i;
	
	for (i = 0; i < rc->nservers; i++) {
	    char *realm = krb_realmofhost (rc->server_list[0].name);
	    if (realm) {
		free (rc->realm);
		rc->realm = strdup (realm);
		if (rc->realm == NULL)
		    return ZERR_BADCONFREALM;
	    }
	}
    }
#endif

    return ZERR_NONE;
}

/*
 * Free all memory that is allocated for `rc'
 */

Code_t
Z_FreeRealmConfig(Z_RealmConfig *rc)
{
    int i;

    for (i = 0; i < rc->nservers; i++)
	free (rc->server_list[i].name);
    free (rc->server_list);
    free (rc);

    return ZERR_NONE;
}
