/* This file is part of the Project Athena Zephyr Notification System.
 * It contains source for the ZMakeAuthentication function.
 *
 *	Created by:	Robert French
 *
 *	Copyright (c) 1987 by the Massachusetts Institute of Technology.
 *	For copying and distribution information, see the file
 *	"mit-copyright.h". 
 */

#include <internal.h>

RCSID("$Id: ZMkAuth.c,v 1.11 2002/12/11 12:32:23 lha Exp $");

#ifdef HAVE_KRB5
#include <krb5.h>
#endif


Code_t ZResetAuthentication () {
#ifdef HAVE_KRB4
    int i;

    for (i=0; i<__nrealms; i++)
	__realm_list[i].last_authent_time = 0;
#endif
    return ZERR_NONE;
}


#ifdef HAVE_KRB5

static int k5z_inited = 0;
static krb5_context k5z_ctx;
static krb5_ccache k5z_id;

static int
_z_init_v5(void)
{
    krb5_error_code ret;

    if (k5z_inited)
	return 0;

    ret = krb5_init_context(&k5z_ctx);
    if (ret)
	return 1;

    ret = krb5_cc_default(k5z_ctx, &k5z_id);
    if (ret) {
	krb5_free_context(k5z_ctx);
	return 1;
    }

    k5z_inited = 1;

    return 0;
}


static int
_z_get_v5(KTEXT authent, char *realm)
{
    krb5_error_code ret;
    krb5_creds in_creds, *real_creds;
    CREDENTIALS c;

    /* 
     * first we check if there exists a 4 ticket, then we don't need
     * to do anything
     */

    ret = krb_get_cred(SERVER_SERVICE, SERVER_INSTANCE, realm, &c);
    if (ret == KSUCCESS) {
	memset(&c, 0, sizeof(c));
	return 0;
    }

    if (_z_init_v5())
	return 1;

    memset(&in_creds, 0, sizeof(in_creds));

    ret = krb5_make_principal(k5z_ctx, &in_creds.server,
			      realm, SERVER_SERVICE, SERVER_INSTANCE,
			      NULL);
    if (ret)
	return 1;

    ret = krb5_cc_get_principal(k5z_ctx, k5z_id, &in_creds.client);
    if (ret) {
	krb5_free_principal(k5z_ctx, in_creds.server);
	return 1;
    }

    in_creds.session.keytype = KEYTYPE_DES;
    
    ret = krb5_get_credentials(k5z_ctx, 0, k5z_id, &in_creds, &real_creds);

    krb5_free_principal(k5z_ctx, in_creds.server);
    krb5_free_principal(k5z_ctx, in_creds.client);
    if (ret)
	return 1;
	
    ret = krb524_convert_creds_kdc_ccache(k5z_ctx, k5z_id, real_creds, &c);
    if (ret)
	return 1;

    krb5_free_creds(k5z_ctx, real_creds);

    tf_init(TKT_FILE, 1);
    tf_save_cred(SERVER_SERVICE, SERVER_INSTANCE, realm,
		 (void*)&c.session, c.lifetime, c.kvno, 
		 &c.ticket_st, c.issue_date);
    tf_close();

    return 0;
}

#endif /* HAVE_KRB5 */


Code_t ZMakeAuthentication(notice, buffer, buffer_len, phdr_len)
    register ZNotice_t *notice;
    char *buffer;
    int buffer_len;
    int *phdr_len;
{
#ifdef HAVE_KRB4
    int i;
    int result;
    time_t now;
    KTEXT_ST authent;
    int cksum_len;
    char *cksum_start, *cstart, *cend;
    ZChecksum_t checksum;
    CREDENTIALS cred;

    if (notice->z_dest_realm) {
	for (i=0; i<__nrealms; i++) {
	    if (strcasecmp(notice->z_dest_realm,
			   __realm_list[i].realm_config.realm) == 0)
		break;
	}
    } else {
	i = 0;
    }

    now = time(0);
    if ((__realm_list[i].last_authent_time == 0) ||
	(now - __realm_list[i].last_authent_time > 120)) {
#if HAVE_KRB5
	_z_get_v5(&authent, ZGetRhs(notice->z_dest_realm));
#endif
	result = krb_mk_req(&authent, SERVER_SERVICE, SERVER_INSTANCE,
			    ZGetRhs(notice->z_dest_realm), 0);
	if (result != MK_AP_OK) {
	    __realm_list[i].last_authent_time = 0;
	    return (result+krb_err_base);
        }
	__realm_list[i].last_authent_time = now;
	__realm_list[i].last_authent = authent;
    }
    else {
	authent = __realm_list[i].last_authent;
    }
    notice->z_auth = 1;
    notice->z_authent_len = authent.length;
    notice->z_ascii_authent = (char *)malloc((unsigned)authent.length*3);
    /* zero length authent is an error, so malloc(0) is not a problem */
    if (!notice->z_ascii_authent)
	return (ENOMEM);
    if ((result = ZMakeAscii(notice->z_ascii_authent, 
			     authent.length*3, 
			     authent.dat, 
			     authent.length)) != ZERR_NONE) {
	free(notice->z_ascii_authent);
	return (result);
    }
    result = Z_FormatRawHeader(notice, buffer, buffer_len, phdr_len,
			       &cksum_start, &cksum_len, &cstart, &cend);
    free(notice->z_ascii_authent);
    notice->z_authent_len = 0;
    if (result)
	return(result);

    /* Compute a checksum over the header and message. */

    if ((result = krb_get_cred(SERVER_SERVICE, SERVER_INSTANCE, 
			       ZGetRhs(notice->z_dest_realm), &cred)) != 0)
	return result;
    checksum = des_quad_cksum(cksum_start, NULL, cstart - cksum_start, 0,
			      cred.session);
    checksum ^= des_quad_cksum(cend, NULL, (cksum_start + cksum_len) - cend, 0,
			       cred.session);
    checksum ^= des_quad_cksum(notice->z_message, NULL, notice->z_message_len,
			       0, cred.session);
    notice->z_checksum = checksum;
    ZMakeAscii32(cstart, (buffer + buffer_len) - cstart, checksum);

    return (ZERR_NONE);
#else
    notice->z_checksum = 0;
    notice->z_auth = 1;
    notice->z_authent_len = 0;
    notice->z_ascii_authent = "";
    return (Z_FormatRawHeader(notice, buffer, buffer_len, phdr_len, &cksum_len,
			      NULL, NULL));
#endif
}
