/* This file is part of the Project Athena Zephyr Notification System.
 * It contains the hostmanager <--> client interaction routines.
 *
 *      Created by:     David C. Jedlinsky
 *
 *      Copyright (c) 1987 by the Massachusetts Institute of Technology.
 *      For copying and distribution information, see the file
 *      "mit-copyright.h". 
 */

#include "zhm.h"

RCSID("$Id: zhm_client.c,v 1.7 2000/06/12 08:18:24 assar Exp $");

extern struct sockaddr_in cli_sin;
extern int noflushflag;

void
transmission_tower(ZNotice_t *notice, struct sockaddr_in *from, 
		   char *packet, int pak_len)
{
    int i;
    realm_info *ri;
    ZNotice_t gack;
    Code_t ret;
    struct sockaddr_in gsin;

    if (notice->z_dest_realm) {
	for (i=0; i<nrealms; i++)
	    if (strcasecmp(realm_list[i].realm_config.realm,
			   notice->z_dest_realm) == 0) {
		ri = &realm_list[i];
		break;
	    }
	if (i == nrealms) {
	    /* XXX I should generate some sort of error here.  Fortunately,
	       only new clients can elicit this error, so I can use a new
	       error value (message body string, probably) here.  For now,
	       just return and let the sender time out. */
	    return;
	}
    } else {
	ri = &realm_list[0];
    }

    if (notice->z_kind == HMCTL) {
	if (!strcmp(notice->z_opcode, CLIENT_FLUSH)) {
	    if (noflushflag)
		syslog(LOG_INFO, "Client requested hm flush (disabled).");
	    else {
		realm_flush(ri);
	    }
	} else if (!strcmp(notice->z_opcode, CLIENT_NEW_SERVER)) {
	    realm_new_server(ri, NULL);
	} else {
	    syslog (LOG_INFO, "Bad control notice from client.");
	}
	return;
    } 

    if (notice->z_kind != UNSAFE) {
	gack = *notice;
	gack.z_kind = HMACK;
	gack.z_message_len = 0;
	gack.z_multinotice = "";
	gsin = cli_sin;
	gsin.sin_port = from->sin_port;
	if (gack.z_port == 0)
	    gack.z_port = from->sin_port;
	notice->z_port = gack.z_port;
	/* Bounce ACK to library */
	if ((ret = send_outgoing(&gsin, &gack)) != ZERR_NONE) {
	    Zperr(ret);
	    com_err("hm", ret, "sending raw notice");
	}
    }

    /* remove the dest realm, since the servers aren't prepared for it */
    notice->z_dest_realm = NULL;

    if (ri->current_server != NO_SERVER) {
	if ((ret = send_outgoing(&ri->sin, notice)) != ZERR_NONE) {
	    Zperr(ret);
	    com_err("hm", ret, "while sending raw notice");
	}
    }

    if (add_notice_to_realm(ri, notice, &gsin, pak_len) != ZERR_NONE)
	syslog(LOG_INFO, "Hey! Insufficient memory to add notice to queue!");
}

Code_t
send_outgoing(struct sockaddr_in *sin, ZNotice_t *notice)
{
    Code_t retval;
    char *packet;
    int length;

    if ((retval = ZSetDestAddr(sin)) != ZERR_NONE)
       return(retval);

    packet = malloc((unsigned)sizeof(ZPacket_t));
    if (packet == NULL)
	return(ENOMEM);

    retval = ZFormatSmallRawNotice(notice, packet, &length);
    if (retval != ZERR_NONE) {
	free(packet);
	return(retval);
    }
    
    retval = ZSendPacket(packet, length, 0);

    free(packet);

    return(retval);
}

