/* This file is part of the Project Athena Zephyr Notification System.
 * It contains source for the ZGetVariable, ZSetVariable, and ZUnsetVariable
 * functions.
 *
 *	Created by:	Robert French
 *
 *	Copyright (c) 1987 by the Massachusetts Institute of Technology.
 *	For copying and distribution information, see the file
 *	"mit-copyright.h". 
 */

#include <internal.h>

RCSID("$Id: ZVariables.c,v 1.11 2000/08/18 15:25:57 assar Exp $");

#include <ctype.h>
#include <pwd.h>

static int get_localvarfile __P((char *bfr, size_t sz));
static const char *get_varval __P((const char *fn, const char *val));
static int varline __P((const char *bfr, const char *var));

const char *
ZGetVariable(const char *var)
{
    char varfile[MAXPATHLEN];
    const char *ret;

    if (get_localvarfile(varfile, sizeof(varfile)))
	return NULL;

    ret = get_varval(varfile, var);
    if (ret != NULL)
	return ret;

    snprintf(varfile, sizeof(varfile), "%s/zephyr.vars", SYSCONFDIR);
    return get_varval(varfile, var);
}

Code_t
ZSetVariable(const char *var, const char *value)
{
    int written;
    FILE *fpin, *fpout;
    char varfile[128], varfilebackup[128], varbfr[512];

    written = 0;
	
    if (get_localvarfile(varfile, sizeof(varfile)))
	return (ZERR_INTERNAL);

    snprintf (varfilebackup, sizeof(varfilebackup),
	      "%s.backup", varfile);
	
    if (!(fpout = fopen(varfilebackup, "w")))
	return (errno);
    if ((fpin = fopen(varfile, "r")) != NULL) {
	while (fgets(varbfr, sizeof varbfr, fpin) != NULL) {
	    if (varbfr[strlen(varbfr)-1] < ' ')
		varbfr[strlen(varbfr)-1] = '\0';
	    if (varline(varbfr, var)) {
		fprintf(fpout, "%s = %s\n", var, value);
		written = 1;
	    }
	    else
		fprintf(fpout, "%s\n", varbfr);
	}
	(void) fclose(fpin);		/* don't care about errs on input */
    } 
    if (!written)
	fprintf(fpout, "%s = %s\n", var, value);
    if (fclose(fpout) == EOF)
	    return(EIO);		/* can't rely on errno */
    if (rename(varfilebackup, varfile))
	return (errno);
    return (ZERR_NONE);
}	

Code_t
ZUnsetVariable(const char *var)
{
    FILE *fpin, *fpout;
    char varfile[128], varfilebackup[128], varbfr[512];

    if (get_localvarfile(varfile, sizeof(varfile)))
	return (ZERR_INTERNAL);

    snprintf (varfilebackup, sizeof(varfilebackup),
	      "%s.backup", varfile);
	
    if (!(fpout = fopen(varfilebackup, "w")))
	return (errno);
    if ((fpin = fopen(varfile, "r")) != NULL) {
	while (fgets(varbfr, sizeof varbfr, fpin) != NULL) {
	    if (varbfr[strlen(varbfr)-1] < ' ')
		varbfr[strlen(varbfr)-1] = '\0';
	    if (!varline(varbfr, var))
		fprintf(fpout, "%s\n", varbfr);
	}
	(void) fclose(fpin);		/* don't care about read close errs */
    } 
    if (fclose(fpout) == EOF)
	    return(EIO);		/* errno isn't reliable */
    if (rename(varfilebackup, varfile))
	return (errno);
    return (ZERR_NONE);
}	

static int
get_localvarfile(char *bfr, size_t sz)
{
    char *path;
    struct passwd *pwd;

    path = getenv("ZEPHYR_VARS");
    if (path == NULL) {
    	path = getenv("HOME");
	if (path == NULL) {
	    if (!(pwd = getpwuid((int) getuid()))) {
		fprintf(stderr, "Zephyr internal failure: "
			"Can't determine your home directory.\n");
		return (1);
	    }
	    path = pwd->pw_dir;
	}
    }
    snprintf (bfr, sz, "%s/.zephyr.vars", path);
    return (0);
} 
	
static const char *
get_varval(const char *fn, const char *var)
{
    FILE *fp;
    static char varbfr[512];
    int i;
	
    fp = fopen(fn, "r");
    if (!fp)
	return NULL;

    while (fgets(varbfr, sizeof varbfr, fp) != NULL) {
	if (varbfr[strlen(varbfr)-1] < ' ')
	    varbfr[strlen(varbfr)-1] = '\0';
	if (!(i = varline(varbfr, var)))
	    continue;
	(void) fclose(fp);		/* open read-only, don't care */
	return (varbfr+i);
    }
    (void) fclose(fp);			/* open read-only, don't care */
    return NULL;
}

/* If the variable in the line bfr[] is the same as var, return index to
   the variable value, else return 0. */
static int
varline(const char *bfr, const char *var)
{
    const char *cp;

    if (!bfr[0] || bfr[0] == '#')	/* comment or null line */
	return (0);
	
    cp = bfr;
    while (*cp && !isspace((unsigned char)(*cp)) && (*cp != '='))
	cp++;

    if (strncasecmp(bfr, var, max(strlen(var),cp - bfr)))
	return(0);			/* var is not the var in
					   bfr ==> no match */

    cp = strchr(bfr, '=');
    if (!cp)
	return(0);
    cp++;
    /* space up to variable value */
    while (*cp && isspace((unsigned char)(*cp)))
	cp++;

    return (cp - bfr);			/* return index */
}
